<?php

namespace App\Console\Commands;

use App\Models\Product;
use App\Models\StockAlert;
use Illuminate\Console\Command;

class CheckStockAlerts extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'stock:check-alerts';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Check for products that reached reorder level and create alerts';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Checking stock levels...');

        // Get products that reached reorder level
        $lowStockProducts = Product::whereColumn('stock_quantity', '<=', 'reorder_level')
            ->where('reorder_level', '>', 0)
            ->get();

        $createdCount = 0;
        $skippedCount = 0;

        foreach ($lowStockProducts as $product) {
            // Check if there's already a pending alert for this product
            $existingAlert = StockAlert::where('product_id', $product->id)
                ->where('status', 'pending')
                ->first();

            if ($existingAlert) {
                // Update existing alert if stock changed
                if ($existingAlert->current_quantity != $product->stock_quantity) {
                    $existingAlert->update([
                        'current_quantity' => $product->stock_quantity,
                    ]);
                    $this->info("Updated alert for product: {$product->name}");
                } else {
                    $skippedCount++;
                }
            } else {
                // Create new alert
                StockAlert::create([
                    'product_id' => $product->id,
                    'current_quantity' => $product->stock_quantity,
                    'reorder_level' => $product->reorder_level,
                    'status' => 'pending',
                    'created_by' => 1, // System user
                ]);
                $createdCount++;
                $this->info("Created alert for product: {$product->name} (Stock: {$product->stock_quantity}, Reorder: {$product->reorder_level})");
            }
        }

        $this->info("Stock check completed. Created: {$createdCount}, Updated/Skipped: {$skippedCount}");
        
        return Command::SUCCESS;
    }
}
