<?php

namespace App\Console\Commands;

use App\Services\LicenseDatabaseService;
use App\Exports\GeneratedLicensesExport;
use Illuminate\Console\Command;
use Illuminate\Support\Str;
use Carbon\Carbon;
use Maatwebsite\Excel\Facades\Excel;

class GenerateLicense extends Command
{
    protected $signature = 'license:generate 
                            {count=1 : عدد التراخيص المراد إنشاؤها}
                            {--email= : Email العميل}
                            {--name= : اسم العميل}
                            {--expires= : تاريخ الانتهاء (Y-m-d) أو null للدائم}
                            {--users=10 : عدد المستخدمين المسموح}
                            {--excel : تصدير التراخيص إلى ملف Excel}';

    protected $description = 'إنشاء ترخيص جديد أو عدة تراخيص في قاعدة البيانات';

    public function handle()
    {
        $count = (int)$this->argument('count');
        
        if ($count < 1) {
            $this->error('عدد التراخيص يجب أن يكون أكبر من 0');
            return 1;
        }

        if ($count > 10000) {
            $this->error('عدد التراخيص يجب أن يكون أقل من 10000');
            return 1;
        }

        $expires = $this->option('expires');
        $maxUsers = (int)($this->option('users') ?? 10);
        $exportToExcel = $this->option('excel');

        // إذا كان عدد واحد فقط، استخدم الطريقة القديمة (مع الأسئلة التفاعلية)
        if ($count === 1 && !$exportToExcel) {
            return $this->generateSingleLicense();
        }

        // إنشاء عدة تراخيص
        $this->info("إنشاء {$count} ترخيص...");
        $this->newLine();

        $licenseDb = app(LicenseDatabaseService::class);
        $bar = $this->output->createProgressBar($count);
        $bar->start();

        $successCount = 0;
        $failedCount = 0;
        $licenses = [];

        // معالجة تاريخ الانتهاء
        $expiresAt = null;
        if ($expires && $expires !== 'null') {
            try {
                $expiresAt = Carbon::parse($expires);
            } catch (\Exception $e) {
                $this->error('تاريخ الانتهاء غير صحيح. استخدم صيغة Y-m-d');
                return 1;
            }
        }

        for ($i = 1; $i <= $count; $i++) {
            // إنشاء License Key
            $licenseKey = strtoupper(
                Str::random(4) . '-' . 
                Str::random(4) . '-' . 
                Str::random(4) . '-' . 
                Str::random(4)
            );

            // إنشاء Serial Number
            $serialNumber = 'SN-' . strtoupper(Str::random(12));

            $licenseData = [
                'license_key' => $licenseKey,
                'serial_number' => $serialNumber,
                'email' => $this->option('email'),
                'customer_name' => $this->option('name') ?? "ترخيص #{$i}",
                'status' => 'pending',
                'expires_at' => $expiresAt ? $expiresAt->format('Y-m-d H:i:s') : null,
                'max_users' => $maxUsers,
                'notes' => "تم إنشاؤه تلقائياً - {$count} تراخيص",
            ];

            if ($licenseDb->insert($licenseData)) {
                $successCount++;
                $licenses[] = [
                    'number' => $i,
                    'license_key' => $licenseKey,
                    'serial_number' => $serialNumber,
                    'expires_at' => $expiresAt ? $expiresAt->format('Y-m-d') : 'دائم / Permanent',
                    'max_users' => $maxUsers,
                ];
            } else {
                $failedCount++;
            }

            $bar->advance();
        }

        $bar->finish();
        $this->newLine(2);

        // عرض النتائج
        $this->info("✅ تم إنشاء {$successCount} ترخيص بنجاح!");
        if ($failedCount > 0) {
            $this->warn("⚠️  فشل إنشاء {$failedCount} ترخيص");
        }

        // تصدير إلى Excel إذا طُلب
        if ($exportToExcel && count($licenses) > 0) {
            $this->exportToExcel($licenses);
        } else {
            // حفظ في ملف نصي
            $this->saveToFile($licenses);
            $this->newLine();
            $this->info("📁 تم حفظ قائمة التراخيص في: storage/app/generated_licenses.txt");
        }

        return 0;
    }

    /**
     * إنشاء ترخيص واحد (الطريقة القديمة)
     */
    protected function generateSingleLicense()
    {
        $this->info('إنشاء ترخيص جديد...');
        $this->newLine();

        // إنشاء License Key
        $licenseKey = strtoupper(
            Str::random(4) . '-' . 
            Str::random(4) . '-' . 
            Str::random(4) . '-' . 
            Str::random(4)
        );

        // إنشاء Serial Number
        $serialNumber = 'SN-' . strtoupper(Str::random(12));

        // معلومات الترخيص
        $email = $this->option('email') ?? $this->ask('Email العميل (اختياري)');
        $name = $this->option('name') ?? $this->ask('اسم العميل (اختياري)');
        $expires = $this->option('expires');
        $maxUsers = (int)($this->option('users') ?? 10);

        if ($expires && $expires !== 'null') {
            try {
                $expiresAt = Carbon::parse($expires);
            } catch (\Exception $e) {
                $this->error('تاريخ الانتهاء غير صحيح. استخدم صيغة Y-m-d');
                return 1;
            }
        } else {
            $expiresAt = null;
        }

        // إنشاء الترخيص في قاعدة البيانات المنفصلة
        $licenseDb = app(LicenseDatabaseService::class);
        
        $licenseData = [
            'license_key' => $licenseKey,
            'serial_number' => $serialNumber,
            'email' => $email,
            'customer_name' => $name,
            'status' => 'pending',
            'expires_at' => $expiresAt ? $expiresAt->format('Y-m-d H:i:s') : null,
            'max_users' => $maxUsers,
        ];
        
        $success = $licenseDb->insert($licenseData);
        
        if (!$success) {
            $this->error('فشل إنشاء الترخيص في قاعدة البيانات');
            return 1;
        }

        $this->newLine();
        $this->info('✅ تم إنشاء الترخيص بنجاح!');
        $this->newLine();
        
        $this->table(
            ['المعلومة', 'القيمة'],
            [
                ['License Key', $licenseKey],
                ['Serial Number', $serialNumber],
                ['Email', $email ?? 'غير محدد'],
                ['اسم العميل', $name ?? 'غير محدد'],
                ['تاريخ الانتهاء', $expiresAt ? $expiresAt->format('Y-m-d') : 'دائم'],
                ['عدد المستخدمين', $maxUsers],
                ['الحالة', 'في انتظار التفعيل'],
                ['Machine ID', 'سيتم حفظه عند التفعيل'],
            ]
        );

        $this->newLine();
        $this->warn('⚠️  مهم: احفظ هذه المعلومات في مكان آمن!');
        $this->warn('⚠️  ستحتاج إلى إعطاء License Key و Serial Number للعميل لتفعيل النظام.');

        return 0;
    }

    /**
     * تصدير التراخيص إلى ملف Excel
     */
    protected function exportToExcel(array $licenses): void
    {
        $this->info('تصدير التراخيص إلى ملف Excel...');
        
        $filename = 'licenses_' . date('Y-m-d_His') . '.xls';
        $filePath = storage_path('app/' . $filename);

        try {
            $export = new GeneratedLicensesExport($licenses);
            
            // تصدير إلى ملف .xls باستخدام Excel::store مع تحديد النوع
            Excel::store($export, $filename, 'local', \Maatwebsite\Excel\Excel::XLS);
            
            // التحقق من وجود الملف
            $actualPath = storage_path('app/' . $filename);
            if (file_exists($actualPath)) {
                $this->newLine();
                $this->info("✅ تم تصدير التراخيص بنجاح!");
                $this->info("📁 الملف: {$actualPath}");
                $this->info("📊 عدد التراخيص: " . count($licenses));
                $this->info("💾 حجم الملف: " . $this->formatBytes(filesize($actualPath)));
            } else {
                // محاولة البحث عن الملف بأسماء مختلفة
                $files = glob(storage_path('app/licenses_*.xls'));
                if (!empty($files)) {
                    $latestFile = max($files);
                    $this->newLine();
                    $this->info("✅ تم تصدير التراخيص بنجاح!");
                    $this->info("📁 الملف: {$latestFile}");
                    $this->info("📊 عدد التراخيص: " . count($licenses));
                    $this->info("💾 حجم الملف: " . $this->formatBytes(filesize($latestFile)));
                } else {
                    $this->warn("⚠️  تم إنشاء الملف ولكن لم يتم العثور عليه في المسار المتوقع.");
                }
            }
        } catch (\Exception $e) {
            $this->error('فشل تصدير الملف: ' . $e->getMessage());
            if ($this->option('verbose')) {
                $this->error('التفاصيل: ' . $e->getTraceAsString());
            }
        }
    }

    /**
     * تنسيق حجم الملف
     */
    protected function formatBytes(int $bytes, int $precision = 2): string
    {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        
        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }
        
        return round($bytes, $precision) . ' ' . $units[$i];
    }

    /**
     * حفظ التراخيص في ملف نصي
     */
    protected function saveToFile(array $licenses): void
    {
        $filePath = storage_path('app/generated_licenses.txt');
        $content = "═══════════════════════════════════════════════════════════════\n";
        $content .= "  قائمة التراخيص المُنشأة\n";
        $content .= "  Generated Licenses List\n";
        $content .= "═══════════════════════════════════════════════════════════════\n\n";
        $content .= "تاريخ الإنشاء: " . now()->format('Y-m-d H:i:s') . "\n";
        $content .= "عدد التراخيص: " . count($licenses) . "\n\n";
        $content .= "═══════════════════════════════════════════════════════════════\n\n";

        foreach ($licenses as $license) {
            $content .= sprintf(
                "ترخيص #%d:\n  License Key: %s\n  Serial Number: %s\n  تاريخ الانتهاء: %s\n  عدد المستخدمين: %d\n\n",
                $license['number'],
                $license['license_key'],
                $license['serial_number'],
                $license['expires_at'],
                $license['max_users']
            );
        }

        file_put_contents($filePath, $content);
    }
}

