<?php

namespace App\Console\Commands;

use App\Models\Category;
use App\Models\Product;
use Illuminate\Console\Command;
use Maatwebsite\Excel\Facades\Excel;

class ImportProductsFromExcel extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'products:import-excel {file=database/products.xls}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Import products and categories from Excel file';

    protected $categoriesCache = [];

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $filePath = $this->argument('file');
        
        if (!file_exists($filePath)) {
            $this->error("File not found: {$filePath}");
            return 1;
        }

        $this->info("Reading Excel file: {$filePath}");
        
        try {
            // Read Excel file as array
            $data = Excel::toArray([], $filePath);
            
            if (empty($data) || empty($data[0])) {
                $this->error("Excel file is empty or invalid.");
                return 1;
            }

            $rows = $data[0];
            $headerRow = array_shift($rows); // Remove header row
            
            $this->info("Found " . count($rows) . " product rows to import");
            
            // First pass: Create all categories
            $this->info("Step 1: Creating categories...");
            $categoriesCreated = $this->createCategories($rows, $headerRow);
            $this->info("Created {$categoriesCreated} categories");
            
            // Second pass: Create products
            $this->info("Step 2: Creating products...");
            $productsCreated = 0;
            $productsSkipped = 0;
            
            $bar = $this->output->createProgressBar(count($rows));
            $bar->start();
            
            foreach ($rows as $rowIndex => $row) {
                $rowData = [];
                foreach ($headerRow as $colIndex => $header) {
                    $rowData[strtolower(trim($header))] = $row[$colIndex] ?? null;
                }
                
                // Skip empty rows
                if (empty($rowData['sku']) && empty($rowData['nom'])) {
                    $bar->advance();
                    continue;
                }
                
                try {
                    $product = $this->createProduct($rowData);
                    if ($product) {
                        $productsCreated++;
                    } else {
                        $productsSkipped++;
                    }
                } catch (\Exception $e) {
                    $this->newLine();
                    $this->warn("Row " . ($rowIndex + 2) . ": " . $e->getMessage());
                    $productsSkipped++;
                }
                
                $bar->advance();
            }
            
            $bar->finish();
            $this->newLine();
            
            $this->info("Import completed!");
            $this->info("Products created: {$productsCreated}");
            $this->info("Products skipped: {$productsSkipped}");
            
            return 0;
            
        } catch (\Exception $e) {
            $this->error("Error importing products: " . $e->getMessage());
            $this->error($e->getTraceAsString());
            return 1;
        }
    }

    /**
     * Create categories from product rows
     */
    protected function createCategories($rows, $headerRow)
    {
        $categoriesCreated = 0;
        $categoryStrings = [];
        
        // Collect all unique category strings
        foreach ($rows as $row) {
            $rowData = [];
            foreach ($headerRow as $colIndex => $header) {
                $rowData[strtolower(trim($header))] = $row[$colIndex] ?? null;
            }
            
            $categoryString = $rowData['category'] ?? null;
            if (!empty($categoryString)) {
                $categoryStrings[] = trim($categoryString);
            }
        }
        
        $categoryStrings = array_unique($categoryStrings);
        
        // Create categories (handle hierarchical structure)
        foreach ($categoryStrings as $categoryString) {
            $this->createCategoryFromString($categoryString, $categoriesCreated);
        }
        
        return $categoriesCreated;
    }

    /**
     * Create category from hierarchical string (e.g., "Parent > Child")
     */
    protected function createCategoryFromString($categoryString, &$count)
    {
        if (empty($categoryString)) {
            return null;
        }
        
        // Check cache first
        if (isset($this->categoriesCache[$categoryString])) {
            return $this->categoriesCache[$categoryString];
        }
        
        // Handle hierarchical categories (e.g., "Cartes de développement > Arduino")
        $parts = explode('>', $categoryString);
        $parts = array_map('trim', $parts);
        
        $parentId = null;
        $categoryName = null;
        
        if (count($parts) > 1) {
            // Has parent category
            $parentName = $parts[0];
            $categoryName = $parts[1];
            
            // Create or get parent category
            $parent = Category::firstOrCreate(
                ['name' => $parentName],
                ['name' => $parentName]
            );
            
            if (!isset($this->categoriesCache[$parentName])) {
                $this->categoriesCache[$parentName] = $parent->id;
                if ($parent->wasRecentlyCreated) {
                    $count++;
                }
            }
            
            $parentId = $parent->id;
        } else {
            // Single level category
            $categoryName = $parts[0];
        }
        
        // Create or get the category (use leaf name, not full path)
        $category = Category::firstOrCreate(
            ['name' => $categoryName],
            ['name' => $categoryName]
        );
        
        $this->categoriesCache[$categoryString] = $category->id;
        
        // Only count if it's a new category
        if ($category->wasRecentlyCreated) {
            $count++;
        }
        
        return $category->id;
    }

    /**
     * Create a product from row data
     */
    protected function createProduct($rowData)
    {
        // Map Excel columns to database fields
        $sku = trim($rowData['sku'] ?? '');
        $name = trim($rowData['nom'] ?? '');
        
        // Skip if no SKU or name
        if (empty($sku) && empty($name)) {
            return null;
        }
        
        // Generate SKU if missing
        if (empty($sku)) {
            $sku = 'SKU-' . strtoupper(substr(md5($name . time()), 0, 8));
        }
        
        // Check if product already exists
        if (Product::where('sku', $sku)->exists()) {
            return null; // Skip duplicates
        }
        
        // Parse numeric values
        $stockQuantity = (int)$this->parseNumeric($rowData['qt_stoc'] ?? $rowData['stock_quantity'] ?? 0);
        $reorderLevel = (int)$this->parseNumeric($rowData['qt_stock_min'] ?? $rowData['reorder_level'] ?? 0);
        
        // Handle price columns - might be in different positions
        $purchasePrice = 0;
        $sellingPrice = 0;
        
        // Try to find price columns
        if (isset($rowData['price'])) {
            $purchasePrice = $this->parseNumeric($rowData['price']);
        }
        
        // Look for selling price (might be in a second price column or calculated)
        // If only one price, use it for both
        if ($purchasePrice > 0 && $sellingPrice == 0) {
            // Default: selling price is 20% more than purchase price
            $sellingPrice = $purchasePrice * 1.2;
        }
        
        // Try to find a second price column (might be unnamed or have different name)
        // Check all numeric columns that might be prices
        foreach ($rowData as $key => $value) {
            if (is_numeric($value) && $value > 0 && $key !== 'qt_stoc' && $key !== 'qt_stock_min') {
                if ($purchasePrice == 0) {
                    $purchasePrice = $this->parseNumeric($value);
                } elseif ($sellingPrice == 0 && $this->parseNumeric($value) != $purchasePrice) {
                    $sellingPrice = $this->parseNumeric($value);
                }
            }
        }
        
        // If still no prices, skip or set defaults
        if ($purchasePrice <= 0) {
            $purchasePrice = 0;
            $sellingPrice = 0;
        }
        
        // Get category ID
        $categoryId = null;
        if (!empty($rowData['category'])) {
            $categoryId = $this->getCategoryId($rowData['category']);
        }
        
        // Create product
        $productData = [
            'sku' => $sku,
            'name' => $name,
            'purchase_price' => $purchasePrice,
            'selling_price' => $sellingPrice > 0 ? $sellingPrice : $purchasePrice,
            'stock_quantity' => $stockQuantity,
            'reorder_level' => $reorderLevel,
            'category_id' => $categoryId,
            'unit' => 'pcs',
            'tax_rate' => 0,
            'description' => null,
        ];
        
        return Product::create($productData);
    }

    /**
     * Get category ID from category string
     */
    protected function getCategoryId($categoryString)
    {
        if (empty($categoryString)) {
            return null;
        }
        
        // Check cache
        if (isset($this->categoriesCache[$categoryString])) {
            return $this->categoriesCache[$categoryString];
        }
        
        // Handle hierarchical categories
        $parts = explode('>', $categoryString);
        $parts = array_map('trim', $parts);
        
        // Use the last part (leaf category) or the full string
        $categoryName = count($parts) > 1 ? $parts[count($parts) - 1] : $parts[0];
        
        $category = Category::where('name', $categoryName)->first();
        
        if ($category) {
            $this->categoriesCache[$categoryString] = $category->id;
            return $category->id;
        }
        
        return null;
    }

    /**
     * Parse numeric value
     */
    protected function parseNumeric($value)
    {
        if (is_numeric($value)) {
            return (float)$value;
        }
        
        // Remove any non-numeric characters except decimal point
        $cleaned = preg_replace('/[^0-9.]/', '', (string)$value);
        return !empty($cleaned) ? (float)$cleaned : 0;
    }
}
