<?php

namespace App\Console\Commands;

use App\Models\Product;
use App\Models\StockTake;
use App\Models\StockTakeItem;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class ScheduleStockTakes extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'stock:schedule-takes {--type=full : Type of stock take (full, partial)}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Schedule periodic stock takes (creates draft stock take records)';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $type = $this->option('type') ?? 'full';
        
        if (!in_array($type, ['full', 'partial'])) {
            $this->error('Invalid type. Use "full" or "partial".');
            return Command::FAILURE;
        }

        $this->info("Creating {$type} stock take...");

        DB::beginTransaction();
        try {
            // Create stock take
            $stockTake = StockTake::create([
                'stock_take_date' => now(),
                'status' => 'draft',
                'type' => $type,
                'notes' => "Scheduled {$type} stock take",
                'created_by' => 1, // System user
            ]);

            // Get products to include
            if ($type === 'full') {
                $products = Product::all();
            } else {
                // Partial: only products with low stock or high movement
                $products = Product::whereColumn('stock_quantity', '<=', 'reorder_level')
                    ->orWhereHas('stockMovements', function ($q) {
                        $q->where('created_at', '>=', now()->subDays(30));
                    })
                    ->get();
            }

            $itemsCreated = 0;
            foreach ($products as $product) {
                StockTakeItem::create([
                    'stock_take_id' => $stockTake->id,
                    'product_id' => $product->id,
                    'system_quantity' => $product->stock_quantity,
                    'counted_quantity' => $product->stock_quantity, // Initially same as system
                    'unit_cost' => $product->purchase_price,
                ]);
                $itemsCreated++;
            }

            DB::commit();
            $this->info("Stock take created: {$stockTake->reference_number} with {$itemsCreated} items.");
            $this->info("Stock take is in draft status. Please complete it manually.");
            
            return Command::SUCCESS;
        } catch (\Exception $e) {
            DB::rollBack();
            $this->error("Error creating stock take: " . $e->getMessage());
            return Command::FAILURE;
        }
    }
}
