<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Symfony\Component\Process\Process;
use Symfony\Component\Process\Exception\ProcessFailedException;

class BackupController extends Controller
{
    protected $backupPath;

    public function __construct()
    {
        $this->backupPath = storage_path('app/backups');
        
        // Ensure backup directory exists
        if (!file_exists($this->backupPath)) {
            mkdir($this->backupPath, 0755, true);
        }
    }

    public function index()
    {
        // Check permissions - only Super Admin
        $user = auth()->user();
        if (!$user->hasRole('Super Admin')) {
            abort(403, trans('messages.unauthorized_action'));
        }

        $backups = $this->getBackups();
        $totalSize = $this->getTotalBackupSize($backups);
        $latestBackup = $backups->first();

        return view('backups.index', compact(
            'backups', 
            'totalSize', 
            'latestBackup'
        ));
    }

    public function store(Request $request)
    {
        // Check permissions - only Super Admin
        $user = auth()->user();
        if (!$user->hasRole('Super Admin')) {
            abort(403, trans('messages.unauthorized_action'));
        }

        try {
            set_time_limit(0); // No timeout
            
            $filename = 'backup-' . date('Y-m-d-H-i-s') . '.sql';
            $filepath = $this->backupPath . '/' . $filename;

            $database = config('database.connections.mysql.database');
            $username = config('database.connections.mysql.username');
            $password = config('database.connections.mysql.password');
            $host = config('database.connections.mysql.host');
            $port = config('database.connections.mysql.port', 3306);

            // Find mysqldump path (Windows XAMPP)
            $mysqldumpPath = '';
            if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
                // Try common XAMPP paths
                $possiblePaths = [
                    'C:\\xampp\\mysql\\bin\\mysqldump.exe',
                    'C:\\xampp\\mysql\\bin\\mysqldump',
                    'mysqldump.exe',
                    'mysqldump',
                ];
                foreach ($possiblePaths as $path) {
                    if (file_exists($path) || shell_exec("where $path 2>nul")) {
                        $mysqldumpPath = $path;
                        break;
                    }
                }
                if (empty($mysqldumpPath)) {
                    $mysqldumpPath = 'mysqldump';
                }
            } else {
                $mysqldumpPath = 'mysqldump';
            }

            // Build mysqldump command
            $command = sprintf(
                '%s --host=%s --port=%s --user=%s --password=%s %s > %s',
                escapeshellarg($mysqldumpPath),
                escapeshellarg($host),
                escapeshellarg($port),
                escapeshellarg($username),
                escapeshellarg($password),
                escapeshellarg($database),
                escapeshellarg($filepath)
            );

            // Execute backup
            $process = Process::fromShellCommandline($command);
            $process->setTimeout(null);
            $process->run();

            if (!$process->isSuccessful()) {
                throw new ProcessFailedException($process);
            }

            // Log backup creation
            $logMessage = 'Backup created: ' . $filename . ' by user: ' . auth()->user()->name . ' (ID: ' . auth()->id() . ')';
            file_put_contents(storage_path('logs/backup.log'), date('Y-m-d H:i:s') . ' - ' . $logMessage . PHP_EOL, FILE_APPEND);

            return redirect()->route('backups.index')
                ->with('success', trans('messages.backup_created_successfully', ['filename' => $filename]));

        } catch (\Exception $e) {
            $logMessage = 'Backup creation failed: ' . $e->getMessage() . ' by user: ' . auth()->user()->name . ' (ID: ' . auth()->id() . ')';
            file_put_contents(storage_path('logs/backup.log'), date('Y-m-d H:i:s') . ' - ERROR - ' . $logMessage . PHP_EOL, FILE_APPEND);

            return redirect()->route('backups.index')
                ->with('error', trans('messages.backup_creation_failed') . ': ' . $e->getMessage());
        }
    }

    public function download($filename)
    {
        // Check permissions - only Super Admin
        $user = auth()->user();
        if (!$user->hasRole('Super Admin')) {
            abort(403, trans('messages.unauthorized_action'));
        }

        // Validate filename to prevent directory traversal
        $filename = basename($filename);
        if (!preg_match('/^backup-\d{4}-\d{2}-\d{2}-\d{2}-\d{2}-\d{2}\.sql$/', $filename)) {
            abort(400, trans('messages.invalid_backup_filename'));
        }

        $filepath = $this->backupPath . '/' . $filename;

        if (!file_exists($filepath)) {
            abort(404, trans('messages.backup_not_found'));
        }

        // Log download
        $logMessage = 'Backup downloaded: ' . $filename . ' by user: ' . auth()->user()->name . ' (ID: ' . auth()->id() . ')';
        file_put_contents(storage_path('logs/backup.log'), date('Y-m-d H:i:s') . ' - ' . $logMessage . PHP_EOL, FILE_APPEND);

        return response()->download($filepath, $filename, [
            'Content-Type' => 'application/sql',
        ]);
    }

    public function destroy($filename)
    {
        // Check permissions - only Super Admin
        $user = auth()->user();
        if (!$user->hasRole('Super Admin')) {
            abort(403, trans('messages.unauthorized_action'));
        }

        // Validate filename to prevent directory traversal
        $filename = basename($filename);
        if (!preg_match('/^backup-\d{4}-\d{2}-\d{2}-\d{2}-\d{2}-\d{2}\.sql$/', $filename)) {
            abort(400, trans('messages.invalid_backup_filename'));
        }

        $filepath = $this->backupPath . '/' . $filename;

        if (!file_exists($filepath)) {
            return redirect()->route('backups.index')
                ->with('error', trans('messages.backup_not_found'));
        }

        try {
            unlink($filepath);

            // Log deletion
            $logMessage = 'Backup deleted: ' . $filename . ' by user: ' . auth()->user()->name . ' (ID: ' . auth()->id() . ')';
            file_put_contents(storage_path('logs/backup.log'), date('Y-m-d H:i:s') . ' - ' . $logMessage . PHP_EOL, FILE_APPEND);

            return redirect()->route('backups.index')
                ->with('success', trans('messages.backup_deleted_successfully'));

        } catch (\Exception $e) {
            $logMessage = 'Backup deletion failed: ' . $filename . ' - ' . $e->getMessage() . ' by user: ' . auth()->user()->name . ' (ID: ' . auth()->id() . ')';
            file_put_contents(storage_path('logs/backup.log'), date('Y-m-d H:i:s') . ' - ERROR - ' . $logMessage . PHP_EOL, FILE_APPEND);

            return redirect()->route('backups.index')
                ->with('error', trans('messages.backup_deletion_failed') . ': ' . $e->getMessage());
        }
    }

    protected function getBackups()
    {
        $files = [];
        $backupPath = $this->backupPath;

        if (is_dir($backupPath)) {
            $items = scandir($backupPath);
            foreach ($items as $item) {
                if ($item !== '.' && $item !== '..' && is_file($backupPath . '/' . $item)) {
                    $filepath = $backupPath . '/' . $item;
                    $files[] = [
                        'filename' => $item,
                        'size' => filesize($filepath),
                        'created_at' => filemtime($filepath),
                    ];
                }
            }
        }

        // Sort by creation date (newest first)
        usort($files, function($a, $b) {
            return $b['created_at'] - $a['created_at'];
        });

        return collect($files);
    }

    protected function getTotalBackupSize($backups)
    {
        return $backups->sum('size');
    }

    public static function createBackup()
    {
        $backupPath = storage_path('app/backups');
        
        if (!file_exists($backupPath)) {
            mkdir($backupPath, 0755, true);
        }

        $filename = 'backup-' . date('Y-m-d-H-i-s') . '.sql';
        $filepath = $backupPath . '/' . $filename;

        $database = config('database.connections.mysql.database');
        $username = config('database.connections.mysql.username');
        $password = config('database.connections.mysql.password');
        $host = config('database.connections.mysql.host');
        $port = config('database.connections.mysql.port', 3306);

        // Find mysqldump path (Windows XAMPP)
        $mysqldumpPath = '';
        if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
            // Try common XAMPP paths
            $possiblePaths = [
                'C:\\xampp\\mysql\\bin\\mysqldump.exe',
                'C:\\xampp\\mysql\\bin\\mysqldump',
                'mysqldump.exe',
                'mysqldump',
            ];
            foreach ($possiblePaths as $path) {
                if (file_exists($path) || shell_exec("where $path 2>nul")) {
                    $mysqldumpPath = $path;
                    break;
                }
            }
            if (empty($mysqldumpPath)) {
                $mysqldumpPath = 'mysqldump';
            }
        } else {
            $mysqldumpPath = 'mysqldump';
        }

        $command = sprintf(
            '%s --host=%s --port=%s --user=%s --password=%s %s > %s',
            escapeshellarg($mysqldumpPath),
            escapeshellarg($host),
            escapeshellarg($port),
            escapeshellarg($username),
            escapeshellarg($password),
            escapeshellarg($database),
            escapeshellarg($filepath)
        );

        $process = Process::fromShellCommandline($command);
        $process->setTimeout(null);
        $process->run();

        if (!$process->isSuccessful()) {
            throw new ProcessFailedException($process);
        }

        $logMessage = 'Scheduled backup created: ' . $filename;
        file_put_contents(storage_path('logs/backup.log'), date('Y-m-d H:i:s') . ' - ' . $logMessage . PHP_EOL, FILE_APPEND);

        return $filepath;
    }

    public static function cleanOldBackups($retentionDays = null)
    {
        if ($retentionDays === null) {
            $retentionDays = \App\Models\Setting::get('backup_retention_days', env('BACKUP_RETENTION_DAYS', 14));
        }
        $retentionDays = (int)$retentionDays;
        $backupPath = storage_path('app/backups');
        $cutoffTime = time() - ($retentionDays * 24 * 60 * 60);

        if (!is_dir($backupPath)) {
            return;
        }

        $items = scandir($backupPath);
        $deletedCount = 0;

        foreach ($items as $item) {
            if ($item !== '.' && $item !== '..' && is_file($backupPath . '/' . $item)) {
                $filepath = $backupPath . '/' . $item;
                $fileTime = filemtime($filepath);

                if ($fileTime < $cutoffTime) {
                    unlink($filepath);
                    $deletedCount++;
                }
            }
        }

        $logMessage = 'Old backups cleaned: ' . $deletedCount . ' backup(s) deleted (retention: ' . $retentionDays . ' days)';
        file_put_contents(storage_path('logs/backup.log'), date('Y-m-d H:i:s') . ' - ' . $logMessage . PHP_EOL, FILE_APPEND);

        return $deletedCount;
    }

    public function exportToSqlite(Request $request)
    {
        // Check permissions - only Super Admin
        $user = auth()->user();
        if (!$user->hasRole('Super Admin')) {
            abort(403, trans('messages.unauthorized_action'));
        }

        try {
            set_time_limit(0); // No timeout
            
            $filename = 'backup-' . date('Y-m-d-H-i-s') . '.sqlite';
            $filepath = $this->backupPath . '/' . $filename;

            // Delete existing SQLite file if exists
            if (file_exists($filepath)) {
                unlink($filepath);
            }

            // Create new SQLite database
            $sqlite = new \PDO('sqlite:' . $filepath);
            $sqlite->setAttribute(\PDO::ATTR_ERRMODE, \PDO::ERRMODE_EXCEPTION);

            // Get all tables from MySQL
            $tables = DB::select('SHOW TABLES');
            $databaseName = config('database.connections.mysql.database');
            $tableKey = 'Tables_in_' . $databaseName;

            foreach ($tables as $table) {
                $tableName = $table->$tableKey;
                
                // Skip migrations table if needed
                if ($tableName === 'migrations') {
                    continue;
                }

                // Get table structure
                $createTable = DB::select("SHOW CREATE TABLE `{$tableName}`");
                $createTableSql = $createTable[0]->{'Create Table'};
                
                // Convert MySQL CREATE TABLE to SQLite compatible
                $createTableSql = $this->convertMySQLToSQLite($createTableSql);
                
                // Create table in SQLite
                $sqlite->exec($createTableSql);

                // Get all data from MySQL table
                $data = DB::table($tableName)->get();
                
                if ($data->count() > 0) {
                    // Get column names
                    $columns = array_keys((array)$data->first());
                    $columnsStr = implode(', ', array_map(function($col) {
                        return '"' . $col . '"';
                    }, $columns));
                    
                    // Prepare insert statement (SQLite uses double quotes, not backticks)
                    $placeholders = implode(', ', array_fill(0, count($columns), '?'));
                    $insertSql = "INSERT INTO \"{$tableName}\" ({$columnsStr}) VALUES ({$placeholders})";
                    $stmt = $sqlite->prepare($insertSql);
                    
                    // Insert data
                    foreach ($data as $row) {
                        $values = [];
                        foreach ($columns as $column) {
                            $value = $row->$column;
                            // Handle null values and convert to appropriate types
                            if ($value === null) {
                                $values[] = null;
                            } else {
                                $values[] = $value;
                            }
                        }
                        $stmt->execute($values);
                    }
                }
            }

            // Log export
            $logMessage = 'SQLite backup created: ' . $filename . ' by user: ' . auth()->user()->name . ' (ID: ' . auth()->id() . ')';
            file_put_contents(storage_path('logs/backup.log'), date('Y-m-d H:i:s') . ' - ' . $logMessage . PHP_EOL, FILE_APPEND);

            return response()->download($filepath, $filename, [
                'Content-Type' => 'application/x-sqlite3',
            ])->deleteFileAfterSend(false);

        } catch (\Exception $e) {
            $logMessage = 'SQLite export failed: ' . $e->getMessage() . ' by user: ' . auth()->user()->name . ' (ID: ' . auth()->id() . ')';
            file_put_contents(storage_path('logs/backup.log'), date('Y-m-d H:i:s') . ' - ERROR - ' . $logMessage . PHP_EOL, FILE_APPEND);

            return redirect()->route('backups.index')
                ->with('error', trans('messages.sqlite_export_failed') . ': ' . $e->getMessage());
        }
    }

    protected function convertMySQLToSQLite($sql)
    {
        // Remove AUTO_INCREMENT from column definitions (must be done first)
        $sql = preg_replace('/\s+AUTO_INCREMENT\s*/i', ' ', $sql);
        $sql = preg_replace('/AUTO_INCREMENT\s+/i', '', $sql);
        
        // Remove MySQL-specific syntax from table definition
        $sql = preg_replace('/AUTO_INCREMENT\s*=\s*\d+/i', '', $sql);
        $sql = preg_replace('/ENGINE\s*=\s*\w+/i', '', $sql);
        $sql = preg_replace('/DEFAULT\s+CHARSET\s*=\s*\w+/i', '', $sql);
        $sql = preg_replace('/COLLATE\s*=\s*\w+/i', '', $sql);
        $sql = preg_replace('/COMMENT\s*=\s*\'[^\']*\'/i', '', $sql);
        $sql = preg_replace('/CHARACTER\s+SET\s+\w+/i', '', $sql);
        
        // Remove UNSIGNED (SQLite doesn't support it)
        $sql = preg_replace('/\s+UNSIGNED\s*/i', ' ', $sql);
        $sql = preg_replace('/\s+UNSIGNED$/i', '', $sql);
        
        // Convert data types (must be done after removing UNSIGNED)
        $sql = preg_replace('/\bTINYINT\s*\([^)]*\)/i', 'INTEGER', $sql);
        $sql = preg_replace('/\bSMALLINT\s*\([^)]*\)/i', 'INTEGER', $sql);
        $sql = preg_replace('/\bMEDIUMINT\s*\([^)]*\)/i', 'INTEGER', $sql);
        $sql = preg_replace('/\bBIGINT\s*\([^)]*\)/i', 'INTEGER', $sql);
        $sql = preg_replace('/\bINT\s*\([^)]*\)/i', 'INTEGER', $sql);
        $sql = preg_replace('/\bTINYINT\b/i', 'INTEGER', $sql);
        $sql = preg_replace('/\bSMALLINT\b/i', 'INTEGER', $sql);
        $sql = preg_replace('/\bMEDIUMINT\b/i', 'INTEGER', $sql);
        $sql = preg_replace('/\bBIGINT\b/i', 'INTEGER', $sql);
        $sql = preg_replace('/\bINT\b/i', 'INTEGER', $sql);
        $sql = preg_replace('/\bTEXT\b/i', 'TEXT', $sql);
        $sql = preg_replace('/\bVARCHAR\s*\([^)]+\)/i', 'TEXT', $sql);
        $sql = preg_replace('/\bCHAR\s*\([^)]+\)/i', 'TEXT', $sql);
        $sql = preg_replace('/\bDATETIME\b/i', 'TEXT', $sql);
        $sql = preg_replace('/\bTIMESTAMP\b/i', 'TEXT', $sql);
        $sql = preg_replace('/\bDATE\b/i', 'TEXT', $sql);
        $sql = preg_replace('/\bTIME\b/i', 'TEXT', $sql);
        $sql = preg_replace('/\bDECIMAL\s*\([^)]+\)/i', 'REAL', $sql);
        $sql = preg_replace('/\bDOUBLE\s*\([^)]*\)/i', 'REAL', $sql);
        $sql = preg_replace('/\bDOUBLE\b/i', 'REAL', $sql);
        $sql = preg_replace('/\bFLOAT\s*\([^)]*\)/i', 'REAL', $sql);
        $sql = preg_replace('/\bFLOAT\b/i', 'REAL', $sql);
        $sql = preg_replace('/\bBOOLEAN\b/i', 'INTEGER', $sql);
        $sql = preg_replace('/\bENUM\s*\([^)]+\)/i', 'TEXT', $sql);
        
        // Remove backticks but keep table and column names
        $sql = preg_replace('/`([^`]+)`/', '$1', $sql);
        
        // Remove DEFAULT CURRENT_TIMESTAMP and similar
        $sql = preg_replace('/DEFAULT\s+CURRENT_TIMESTAMP[^,\s)]*/i', '', $sql);
        $sql = preg_replace('/ON\s+UPDATE\s+CURRENT_TIMESTAMP[^,\s)]*/i', '', $sql);
        
        // Remove ZEROFILL
        $sql = preg_replace('/\s+ZEROFILL\s*/i', ' ', $sql);
        
        // Clean up extra commas and spaces
        $sql = preg_replace('/,\s*\)/', ')', $sql);
        $sql = preg_replace('/\s+/', ' ', $sql);
        $sql = preg_replace('/\s*,\s*/', ', ', $sql);
        $sql = preg_replace('/\s*\(\s*/', '(', $sql);
        $sql = preg_replace('/\s*\)\s*/', ')', $sql);
        
        return trim($sql);
    }

    public function restore(Request $request, $filename)
    {
        // Check permissions - only Super Admin
        $user = auth()->user();
        if (!$user->hasRole('Super Admin')) {
            abort(403, trans('messages.unauthorized_action'));
        }

        // Validate filename to prevent directory traversal
        $filename = basename($filename);
        if (!preg_match('/^backup-\d{4}-\d{2}-\d{2}-\d{2}-\d{2}-\d{2}\.sql$/', $filename)) {
            abort(400, trans('messages.invalid_backup_filename'));
        }

        $filepath = $this->backupPath . '/' . $filename;

        if (!file_exists($filepath)) {
            abort(404, trans('messages.backup_not_found'));
        }

        try {
            set_time_limit(0); // No timeout

            $database = config('database.connections.mysql.database');
            $username = config('database.connections.mysql.username');
            $password = config('database.connections.mysql.password');
            $host = config('database.connections.mysql.host');
            $port = config('database.connections.mysql.port', 3306);

            // Find mysql path (Windows XAMPP)
            $mysqlPath = '';
            if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
                $possiblePaths = [
                    'C:\\xampp\\mysql\\bin\\mysql.exe',
                    'C:\\xampp\\mysql\\bin\\mysql',
                    'mysql.exe',
                    'mysql',
                ];
                foreach ($possiblePaths as $path) {
                    if (file_exists($path) || shell_exec("where $path 2>nul")) {
                        $mysqlPath = $path;
                        break;
                    }
                }
                if (empty($mysqlPath)) {
                    $mysqlPath = 'mysql';
                }
            } else {
                $mysqlPath = 'mysql';
            }

            // Use DB::unprepared method for better compatibility
            // Read SQL file content
            $sqlContent = file_get_contents($filepath);
            
            if ($sqlContent === false || empty($sqlContent)) {
                throw new \Exception('Failed to read backup file or file is empty');
            }

            // Remove BOM if present
            $sqlContent = preg_replace('/^\xEF\xBB\xBF/', '', $sqlContent);
            
            // Disable foreign key checks temporarily
            DB::statement('SET FOREIGN_KEY_CHECKS=0;');
            DB::statement('SET SQL_MODE="NO_AUTO_VALUE_ON_ZERO";');
            
            try {
                // Split SQL into statements
                // First, remove comments
                $sqlContent = preg_replace('/--.*$/m', '', $sqlContent);
                $sqlContent = preg_replace('/\/\*.*?\*\//s', '', $sqlContent);
                
                // Split by semicolon, but be careful with strings
                $statements = [];
                $current = '';
                $inString = false;
                $stringChar = '';
                $escaped = false;
                
                for ($i = 0; $i < strlen($sqlContent); $i++) {
                    $char = $sqlContent[$i];
                    
                    if ($escaped) {
                        $current .= $char;
                        $escaped = false;
                        continue;
                    }
                    
                    if ($char === '\\') {
                        $escaped = true;
                        $current .= $char;
                        continue;
                    }
                    
                    if (!$inString && ($char === '"' || $char === "'" || $char === '`')) {
                        $inString = true;
                        $stringChar = $char;
                        $current .= $char;
                    } elseif ($inString && $char === $stringChar) {
                        $inString = false;
                        $stringChar = '';
                        $current .= $char;
                    } elseif (!$inString && $char === ';') {
                        $stmt = trim($current);
                        if (!empty($stmt) && strlen($stmt) > 3) {
                            // Skip empty or very short statements
                            if (!preg_match('/^(SET|USE|\/\*|\*\/|\-\-)/i', $stmt)) {
                                $statements[] = $stmt;
                            }
                        }
                        $current = '';
                    } else {
                        $current .= $char;
                    }
                }
                
                // Add last statement if exists
                $stmt = trim($current);
                if (!empty($stmt) && strlen($stmt) > 3) {
                    if (!preg_match('/^(SET|USE|\/\*|\*\/|\-\-)/i', $stmt)) {
                        $statements[] = $stmt;
                    }
                }

                // Execute statements in chunks for better performance
                $executed = 0;
                $errors = [];
                $chunkSize = 50; // Execute in chunks
                
                foreach (array_chunk($statements, $chunkSize) as $chunk) {
                    foreach ($chunk as $statement) {
                        if (empty(trim($statement))) {
                            continue;
                        }
                        
                        try {
                            DB::unprepared($statement);
                            $executed++;
                        } catch (\Exception $e) {
                            $errorMsg = substr($e->getMessage(), 0, 200);
                            $errors[] = $errorMsg;
                            Log::warning('Backup restore statement error: ' . $errorMsg);
                            // Continue with other statements
                        }
                    }
                }

                // Re-enable foreign key checks
                DB::statement('SET FOREIGN_KEY_CHECKS=1;');
                DB::statement('SET SQL_MODE="";');

                if ($executed === 0) {
                    throw new \Exception('No statements were executed. Backup file may be invalid or empty.');
                }
                
                if (count($errors) > 0) {
                    Log::warning('Backup restore completed with ' . count($errors) . ' errors. Successfully executed: ' . $executed . ' statements.');
                }
                
            } catch (\Exception $e) {
                // Re-enable foreign key checks even on error
                DB::statement('SET FOREIGN_KEY_CHECKS=1;');
                DB::statement('SET SQL_MODE="";');
                throw $e;
            }

            // Log restore
            $logMessage = 'Backup restored: ' . $filename . ' by user: ' . auth()->user()->name . ' (ID: ' . auth()->id() . ')';
            file_put_contents(storage_path('logs/backup.log'), date('Y-m-d H:i:s') . ' - ' . $logMessage . PHP_EOL, FILE_APPEND);

            return redirect()->route('backups.index')
                ->with('success', trans('messages.backup_restored_successfully', ['filename' => $filename]));

        } catch (\Exception $e) {
            $logMessage = 'Backup restore failed: ' . $filename . ' - ' . $e->getMessage() . ' by user: ' . auth()->user()->name . ' (ID: ' . auth()->id() . ')';
            file_put_contents(storage_path('logs/backup.log'), date('Y-m-d H:i:s') . ' - ERROR - ' . $logMessage . PHP_EOL, FILE_APPEND);

            return redirect()->route('backups.index')
                ->with('error', trans('messages.backup_restore_failed') . ': ' . $e->getMessage());
        }
    }

    public function updateSettings(Request $request)
    {
        // Check permissions - only Super Admin
        $user = auth()->user();
        if (!$user->hasRole('Super Admin')) {
            abort(403, trans('messages.unauthorized_action'));
        }

        $validated = $request->validate([
            'auto_backup_enabled' => 'nullable|in:0,1',
            'auto_backup_frequency' => 'required|in:daily,weekly',
            'auto_backup_time' => 'required|regex:/^([0-1][0-9]|2[0-3]):[0-5][0-9]$/',
            'backup_retention_days' => 'required|integer|min:1|max:365',
        ]);

        \App\Models\Setting::set('auto_backup_enabled', $validated['auto_backup_enabled'] ?? '0');
        \App\Models\Setting::set('auto_backup_frequency', $validated['auto_backup_frequency']);
        \App\Models\Setting::set('auto_backup_time', $validated['auto_backup_time']);
        \App\Models\Setting::set('backup_retention_days', $validated['backup_retention_days']);

        // Update environment variable for retention days
        $envFile = base_path('.env');
        if (file_exists($envFile)) {
            $envContent = file_get_contents($envFile);
            $pattern = '/^BACKUP_RETENTION_DAYS=.*/m';
            $replacement = 'BACKUP_RETENTION_DAYS=' . $validated['backup_retention_days'];
            if (preg_match($pattern, $envContent)) {
                $envContent = preg_replace($pattern, $replacement, $envContent);
            } else {
                $envContent .= PHP_EOL . $replacement;
            }
            file_put_contents($envFile, $envContent);
        }

        return redirect()->route('backups.index')
            ->with('success', trans('messages.backup_settings_updated'));
    }
}
