<?php

namespace App\Http\Controllers;

use App\Models\Cheque;
use App\Models\Customer;
use App\Models\Supplier;
use App\Models\Invoice;
use App\Models\PurchaseInvoice;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ChequeController extends Controller
{
    public function index(Request $request)
    {
        $query = Cheque::with(['customer', 'supplier', 'invoice', 'purchaseInvoice', 'user']);

        // Filter by type
        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by customer
        if ($request->filled('customer_id')) {
            $query->where('customer_id', $request->customer_id);
        }

        // Filter by supplier
        if ($request->filled('supplier_id')) {
            $query->where('supplier_id', $request->supplier_id);
        }

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('cheque_number', 'like', "%{$search}%")
                  ->orWhere('bank_name', 'like', "%{$search}%")
                  ->orWhereHas('customer', function ($q) use ($search) {
                      $q->where('name', 'like', "%{$search}%");
                  })
                  ->orWhereHas('supplier', function ($q) use ($search) {
                      $q->where('name', 'like', "%{$search}%");
                  });
            });
        }

        // Filter by due date
        if ($request->filled('due_date_from')) {
            $query->where('due_date', '>=', $request->due_date_from);
        }
        if ($request->filled('due_date_to')) {
            $query->where('due_date', '<=', $request->due_date_to);
        }

        $query->orderBy('due_date', 'asc');

        $itemsPerPage = Setting::getItemsPerPage(20);
        $cheques = $query->paginate($itemsPerPage)->appends($request->query());

        // Update status to 'due' for cheques that are past due date
        $cheques->getCollection()->each(function ($cheque) {
            if ($cheque->status === 'pending' && $cheque->due_date < now() && !$cheque->isPaid() && !$cheque->isReturned()) {
                $cheque->update(['status' => 'due']);
            }
        });

        $customers = Customer::orderBy('name')->get();
        $suppliers = Supplier::orderBy('name')->get();

        return view('cheques.index', compact('cheques', 'customers', 'suppliers'));
    }

    public function create(Request $request)
    {
        $customers = Customer::orderBy('name')->get();
        $suppliers = Supplier::orderBy('name')->get();
        
        $type = $request->get('type', 'customer');
        $customerId = $request->get('customer_id');
        $supplierId = $request->get('supplier_id');
        $invoiceId = $request->get('invoice_id');
        $purchaseInvoiceId = $request->get('purchase_invoice_id');
        $amount = $request->get('amount');

        $invoices = [];
        $purchaseInvoices = [];

        if ($type === 'customer' && $customerId) {
            $query = Invoice::where('customer_id', $customerId)
                ->whereIn('status', ['final', 'partially_paid', 'paid']);
            
            // If a specific invoice_id is provided, include it even if it's paid
            if ($invoiceId) {
                $query->orWhere('id', $invoiceId);
            }
            
            $invoices = $query->orderBy('created_at', 'desc')->get();
        }

        if ($type === 'supplier' && $supplierId) {
            $query = PurchaseInvoice::where('supplier_id', $supplierId)
                ->whereIn('status', ['confirmed', 'partially_paid', 'paid']);
            
            // If a specific purchase_invoice_id is provided, include it even if it's paid
            if ($purchaseInvoiceId) {
                $query->orWhere('id', $purchaseInvoiceId);
            }
            
            $purchaseInvoices = $query->orderBy('created_at', 'desc')->get();
        }

        return view('cheques.create', compact('customers', 'suppliers', 'type', 'customerId', 'supplierId', 'invoiceId', 'purchaseInvoiceId', 'amount', 'invoices', 'purchaseInvoices'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'type' => 'required|in:customer,supplier',
            'customer_id' => 'required_if:type,customer|nullable|exists:customers,id',
            'supplier_id' => 'required_if:type,supplier|nullable|exists:suppliers,id',
            'invoice_id' => 'nullable|exists:invoices,id',
            'purchase_invoice_id' => 'nullable|exists:purchase_invoices,id',
            'amount' => 'required|numeric|min:0.01',
            'cheque_number' => 'required|string|max:255|unique:cheques,cheque_number',
            'bank_name' => 'required|string|max:255',
            'bank_account' => 'nullable|string|max:255',
            'issue_date' => 'required|date',
            'due_date' => 'required|date|after_or_equal:issue_date',
            'notes' => 'nullable|string',
        ]);

        // Validate that customer_id or supplier_id is provided based on type
        if ($validated['type'] === 'customer' && !$validated['customer_id']) {
            return back()->withErrors(['customer_id' => trans('messages.customer_required')])->withInput();
        }
        if ($validated['type'] === 'supplier' && !$validated['supplier_id']) {
            return back()->withErrors(['supplier_id' => trans('messages.supplier_required')])->withInput();
        }

        DB::beginTransaction();
        try {
            $cheque = Cheque::create([
                'type' => $validated['type'],
                'customer_id' => $validated['customer_id'] ?? null,
                'supplier_id' => $validated['supplier_id'] ?? null,
                'invoice_id' => $validated['invoice_id'] ?? null,
                'purchase_invoice_id' => $validated['purchase_invoice_id'] ?? null,
                'amount' => $validated['amount'],
                'cheque_number' => $validated['cheque_number'],
                'bank_name' => $validated['bank_name'],
                'bank_account' => $validated['bank_account'] ?? null,
                'issue_date' => $validated['issue_date'],
                'due_date' => $validated['due_date'],
                'status' => 'pending',
                'notes' => $validated['notes'] ?? null,
                'user_id' => auth()->id(),
            ]);

            DB::commit();

            return redirect()->route('cheques.index')
                ->with('success', trans('messages.cheque_created_successfully'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.error_creating_cheque') . ': ' . $e->getMessage())->withInput();
        }
    }

    public function show(Cheque $cheque)
    {
        $cheque->load(['customer', 'supplier', 'invoice', 'purchaseInvoice', 'user']);
        return view('cheques.show', compact('cheque'));
    }

    public function edit(Cheque $cheque)
    {
        $customers = Customer::orderBy('name')->get();
        $suppliers = Supplier::orderBy('name')->get();
        
        $invoices = [];
        $purchaseInvoices = [];

        if ($cheque->type === 'customer' && $cheque->customer_id) {
            $invoices = Invoice::where('customer_id', $cheque->customer_id)
                ->whereIn('status', ['final', 'partially_paid'])
                ->orderBy('created_at', 'desc')
                ->get();
        }

        if ($cheque->type === 'supplier' && $cheque->supplier_id) {
            $purchaseInvoices = PurchaseInvoice::where('supplier_id', $cheque->supplier_id)
                ->whereIn('status', ['confirmed', 'partially_paid'])
                ->orderBy('created_at', 'desc')
                ->get();
        }

        return view('cheques.edit', compact('cheque', 'customers', 'suppliers', 'invoices', 'purchaseInvoices'));
    }

    public function update(Request $request, Cheque $cheque)
    {
        $validated = $request->validate([
            'type' => 'required|in:customer,supplier',
            'customer_id' => 'required_if:type,customer|nullable|exists:customers,id',
            'supplier_id' => 'required_if:type,supplier|nullable|exists:suppliers,id',
            'invoice_id' => 'nullable|exists:invoices,id',
            'purchase_invoice_id' => 'nullable|exists:purchase_invoices,id',
            'amount' => 'required|numeric|min:0.01',
            'cheque_number' => 'required|string|max:255|unique:cheques,cheque_number,' . $cheque->id,
            'bank_name' => 'required|string|max:255',
            'bank_account' => 'nullable|string|max:255',
            'issue_date' => 'required|date',
            'due_date' => 'required|date|after_or_equal:issue_date',
            'notes' => 'nullable|string',
        ]);

        if ($validated['type'] === 'customer' && !$validated['customer_id']) {
            return back()->withErrors(['customer_id' => trans('messages.customer_required')])->withInput();
        }
        if ($validated['type'] === 'supplier' && !$validated['supplier_id']) {
            return back()->withErrors(['supplier_id' => trans('messages.supplier_required')])->withInput();
        }

        // Don't allow editing if cheque is paid or returned
        if (in_array($cheque->status, ['paid', 'returned'])) {
            return back()->with('error', trans('messages.cannot_edit_paid_or_returned_cheque'));
        }

        DB::beginTransaction();
        try {
            $cheque->update([
                'type' => $validated['type'],
                'customer_id' => $validated['customer_id'] ?? null,
                'supplier_id' => $validated['supplier_id'] ?? null,
                'invoice_id' => $validated['invoice_id'] ?? null,
                'purchase_invoice_id' => $validated['purchase_invoice_id'] ?? null,
                'amount' => $validated['amount'],
                'cheque_number' => $validated['cheque_number'],
                'bank_name' => $validated['bank_name'],
                'bank_account' => $validated['bank_account'] ?? null,
                'issue_date' => $validated['issue_date'],
                'due_date' => $validated['due_date'],
                'notes' => $validated['notes'] ?? null,
            ]);

            DB::commit();

            return redirect()->route('cheques.index')
                ->with('success', trans('messages.cheque_updated_successfully'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.error_updating_cheque') . ': ' . $e->getMessage())->withInput();
        }
    }

    public function destroy(Cheque $cheque)
    {
        if (in_array($cheque->status, ['paid', 'returned'])) {
            return back()->with('error', trans('messages.cannot_delete_paid_or_returned_cheque'));
        }

        $cheque->delete();

        return redirect()->route('cheques.index')
            ->with('success', trans('messages.cheque_deleted_successfully'));
    }

    public function markAsPaid(Request $request, Cheque $cheque)
    {
        if ($cheque->isPaid()) {
            return back()->with('error', trans('messages.cheque_already_paid'));
        }

        $validated = $request->validate([
            'paid_date' => 'required|date|after_or_equal:' . $cheque->issue_date->format('Y-m-d'),
        ]);

        $cheque->update([
            'status' => 'paid',
            'paid_date' => $validated['paid_date'],
        ]);

        return back()->with('success', trans('messages.cheque_marked_as_paid'));
    }

    public function markAsReturned(Request $request, Cheque $cheque)
    {
        if ($cheque->isPaid()) {
            return back()->with('error', trans('messages.cannot_return_paid_cheque'));
        }

        if ($cheque->isReturned()) {
            return back()->with('error', trans('messages.cheque_already_returned'));
        }

        $validated = $request->validate([
            'returned_date' => 'required|date|after_or_equal:' . $cheque->issue_date->format('Y-m-d'),
            'return_reason' => 'required|string|max:500',
        ]);

        $cheque->update([
            'status' => 'returned',
            'returned_date' => $validated['returned_date'],
            'return_reason' => $validated['return_reason'],
        ]);

        return back()->with('success', trans('messages.cheque_marked_as_returned'));
    }

    public function cancel(Cheque $cheque)
    {
        if (in_array($cheque->status, ['paid', 'returned'])) {
            return back()->with('error', trans('messages.cannot_cancel_paid_or_returned_cheque'));
        }

        $cheque->update([
            'status' => 'cancelled',
        ]);

        return back()->with('success', trans('messages.cheque_cancelled'));
    }

    public function getInvoices($customerId, Request $request)
    {
        $invoiceId = $request->get('invoice_id');
        
        // Build query to include invoices with specific statuses OR the specific invoice_id
        $query = Invoice::where(function($q) use ($customerId, $invoiceId) {
            $q->where('customer_id', $customerId)
              ->whereIn('status', ['final', 'partially_paid', 'paid']);
            
            // If a specific invoice_id is provided, include it even if it's paid or doesn't match customer
            if ($invoiceId) {
                $q->orWhere('id', $invoiceId);
            }
        });
        
        $invoices = $query->orderBy('created_at', 'desc')->get();

        return response()->json($invoices->map(function ($invoice) {
            return [
                'id' => $invoice->id,
                'invoice_number' => $invoice->invoice_number,
                'due_amount' => $invoice->due_amount,
                'formatted_due_amount' => format_currency($invoice->due_amount),
            ];
        }));
    }

    public function getPurchaseInvoices($supplierId, Request $request)
    {
        $purchaseInvoiceId = $request->get('purchase_invoice_id');
        
        // Build query to include purchase invoices with specific statuses OR the specific purchase_invoice_id
        $query = PurchaseInvoice::where(function($q) use ($supplierId, $purchaseInvoiceId) {
            $q->where('supplier_id', $supplierId)
              ->whereIn('status', ['confirmed', 'partially_paid', 'paid']);
            
            // If a specific purchase_invoice_id is provided, include it even if it's paid or doesn't match supplier
            if ($purchaseInvoiceId) {
                $q->orWhere('id', $purchaseInvoiceId);
            }
        });
        
        $purchaseInvoices = $query->orderBy('created_at', 'desc')->get();

        return response()->json($purchaseInvoices->map(function ($purchaseInvoice) {
            return [
                'id' => $purchaseInvoice->id,
                'invoice_number' => $purchaseInvoice->invoice_number,
                'amount_remaining' => $purchaseInvoice->amount_remaining,
                'formatted_amount_remaining' => format_currency($purchaseInvoice->amount_remaining),
            ];
        }));
    }
}
