<?php

namespace App\Http\Controllers;

use App\Models\Customer;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Traits\Sortable;
use Maatwebsite\Excel\Facades\Excel;
use App\Imports\CustomersImport;

class CustomerController extends Controller
{
    use Sortable;

    public function index(Request $request)
    {
        $query = Customer::query();

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('phone', 'like', "%{$search}%");
            });
        }

        // Filter by customers with debts (balance > 0)
        if ($request->filled('filter') && $request->filter === 'with_debts') {
            // We'll filter after calculating balances
        }

        $allowedColumns = ['name', 'email', 'phone', 'created_at'];
        $sort = $request->get('sort', 'created_at');
        $direction = $request->get('direction', 'desc');

        if (in_array($sort, $allowedColumns)) {
            $query->orderBy($sort, $direction);
        } else {
            $query->latest();
        }

        $itemsPerPage = Setting::getItemsPerPage(20);
        $customers = $query->with('invoices')
            ->paginate($itemsPerPage)
            ->appends($request->query());

        // Calculate current balance for each customer
        $customers->getCollection()->transform(function ($customer) {
            $customer->calculated_balance = $customer->calculateCurrentBalance();
            return $customer;
        });

        // Filter by debts after calculating balances
        if ($request->filled('filter') && $request->filter === 'with_debts') {
            $customers->setCollection(
                $customers->getCollection()->filter(function ($customer) {
                    return ($customer->calculated_balance ?? $customer->calculateCurrentBalance()) > 0;
                })
            );
        }

        return view('customers.index', compact('customers', 'sort', 'direction'));
    }

    public function printWithDebts(Request $request)
    {
        $customers = Customer::with('invoices')->get();
        
        // Calculate balance and filter customers with debts
        $customersWithDebts = $customers->map(function ($customer) {
            $customer->calculated_balance = $customer->calculateCurrentBalance();
            return $customer;
        })->filter(function ($customer) {
            return $customer->calculated_balance > 0;
        })->sortByDesc('calculated_balance');

        $totalDebts = $customersWithDebts->sum('calculated_balance');
        $storeName = Setting::get('store_name', 'Gestock Pro');
        $storeLogo = Setting::get('store_logo');
        $locale = app()->getLocale();
        $isRTL = $locale === 'ar';
        $dir = $isRTL ? 'rtl' : 'ltr';

        return view('customers.print-with-debts', compact('customersWithDebts', 'totalDebts', 'storeName', 'storeLogo', 'locale', 'isRTL', 'dir'));
    }

    public function create()
    {
        return view('customers.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'nullable|string|max:50',
            'email' => 'nullable|email|unique:customers,email',
            'address' => 'nullable|string',
            'current_balance' => 'nullable|numeric|min:0',
        ]);

        Customer::create($validated);

        return redirect()->route('customers.index')
            ->with('success', trans('messages.customer_created'));
    }

    public function show(Customer $customer, Request $request)
    {
        $customer->load(['invoices.items.product', 'invoices.payments', 'cheques', 'promissoryNotes']);
        $customer->calculated_balance = $customer->calculateCurrentBalance();

        // Get date filters
        $dateFrom = $request->get('date_from');
        $dateTo = $request->get('date_to');

        // Get all invoices (excluding drafts and returned)
        $invoicesQuery = $customer->invoices()
            ->whereNotIn('status', ['draft', 'returned'])
            ->with(['items.product', 'payments']);

        if ($dateFrom) {
            $invoicesQuery->whereDate('created_at', '>=', $dateFrom);
        }
        if ($dateTo) {
            $invoicesQuery->whereDate('created_at', '<=', $dateTo);
        }

        $invoices = $invoicesQuery->latest()->get();

        // Calculate purchase statistics
        $totalPurchases = $invoices->sum('total');
        $totalPaid = $invoices->sum('paid_amount');
        $totalDue = $invoices->sum('due_amount');
        $invoiceCount = $invoices->count();
        $averagePurchase = $invoiceCount > 0 ? $totalPurchases / $invoiceCount : 0;

        // Calculate profit for customer (revenue - cost)
        $totalRevenue = $invoices->sum('total');
        $totalCost = $invoices->sum(function ($invoice) {
            return $invoice->items->sum(function ($item) {
                return $item->quantity * ($item->product->purchase_price ?? 0);
            });
        });
        $totalProfit = $totalRevenue - $totalCost;
        $profitMargin = $totalRevenue > 0 ? ($totalProfit / $totalRevenue) * 100 : 0;

        // Get top products purchased
        $productPurchases = [];
        foreach ($invoices as $invoice) {
            foreach ($invoice->items as $item) {
                $productId = $item->product_id;
                if (!isset($productPurchases[$productId])) {
                    $productPurchases[$productId] = [
                        'product' => $item->product,
                        'quantity' => 0,
                        'total_amount' => 0,
                        'invoice_count' => 0,
                    ];
                }
                $productPurchases[$productId]['quantity'] += $item->quantity;
                $productPurchases[$productId]['total_amount'] += $item->total;
                $productPurchases[$productId]['invoice_count']++;
            }
        }
        $topProducts = collect($productPurchases)->sortByDesc('total_amount')->take(10)->values();

        // Monthly purchases breakdown
        $monthlyPurchases = $invoices->groupBy(function ($invoice) {
            return $invoice->created_at->format('Y-m');
        })->map(function ($group, $month) {
            return [
                'month' => $month,
                'count' => $group->count(),
                'total' => $group->sum('total'),
                'profit' => $group->sum(function ($inv) {
                    return $inv->items->sum(function ($item) {
                        return $item->total - ($item->quantity * ($item->product->purchase_price ?? 0));
                    });
                }),
            ];
        })->sortBy('month')->values();

        // Payment method breakdown
        $paymentMethods = [];
        foreach ($invoices as $invoice) {
            foreach ($invoice->payments as $payment) {
                $method = $payment->payment_method;
                if (!isset($paymentMethods[$method])) {
                    $paymentMethods[$method] = [
                        'method' => $method,
                        'count' => 0,
                        'total' => 0,
                    ];
                }
                $paymentMethods[$method]['count']++;
                $paymentMethods[$method]['total'] += $payment->amount;
            }
        }
        $paymentMethodsBreakdown = collect($paymentMethods)->sortByDesc('total')->values();

        return view('customers.show', compact(
            'customer',
            'totalPurchases',
            'totalPaid',
            'totalDue',
            'invoiceCount',
            'averagePurchase',
            'totalRevenue',
            'totalCost',
            'totalProfit',
            'profitMargin',
            'topProducts',
            'monthlyPurchases',
            'paymentMethodsBreakdown',
            'dateFrom',
            'dateTo',
            'invoices'
        ));
    }

    public function edit(Customer $customer)
    {
        return view('customers.edit', compact('customer'));
    }

    public function update(Request $request, Customer $customer)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'nullable|string|max:50',
            'email' => ['nullable', 'email', Rule::unique('customers')->ignore($customer->id)],
            'address' => 'nullable|string',
            'current_balance' => 'nullable|numeric',
        ]);

        $customer->update($validated);

        return redirect()->route('customers.index')
            ->with('success', trans('messages.customer_updated'));
    }

    public function destroy(Customer $customer)
    {
        if ($customer->invoices()->count() > 0) {
            return redirect()->route('customers.index')
                ->with('error', trans('messages.cannot_delete_customers_with_invoices'));
        }

        $customer->delete();

        return redirect()->route('customers.index')
            ->with('success', trans('messages.customer_deleted'));
    }

    public function bulkDelete(Request $request)
    {
        // Check permissions - Only Admin and Super Admin can bulk delete
        $user = auth()->user();
        if (!$user->hasAnyRole(['Admin', 'Super Admin'])) {
            abort(403, trans('messages.unauthorized_action'));
        }

        $request->validate([
            'selected_ids' => 'required|array|min:1',
            'selected_ids.*' => 'exists:customers,id',
        ]);

        try {
            $selectedIds = $request->selected_ids;
            $customers = Customer::whereIn('id', $selectedIds)->get();
            
            // Check for customers with invoices
            $customersWithInvoices = $customers->filter(function ($customer) {
                return $customer->invoices()->count() > 0;
            });

            if ($customersWithInvoices->count() > 0) {
                return back()->with('error', trans('messages.cannot_delete_customers_with_invoices'));
            }
            
            // Log before deletion
            foreach ($customers as $customer) {
                \App\Models\AuditLog::create([
                    'action' => 'bulk_delete',
                    'model_type' => Customer::class,
                    'model_id' => $customer->id,
                    'old_values' => $customer->toArray(),
                    'new_values' => null,
                    'user_id' => auth()->id(),
                    'ip_address' => request()->ip(),
                    'user_agent' => request()->userAgent(),
                ]);
            }

            Customer::whereIn('id', $selectedIds)->delete();

            return redirect()->route('customers.index')
                ->with('success', trans('messages.bulk_delete_success', ['count' => count($selectedIds)]));
        } catch (\Exception $e) {
            return back()->with('error', trans('messages.bulk_delete_error') . ': ' . $e->getMessage());
        }
    }

    public function importForm()
    {
        // Check permissions - Only Admin and Super Admin can import
        $user = auth()->user();
        if (!$user->hasAnyRole(['Admin', 'Super Admin'])) {
            abort(403, trans('messages.unauthorized_action'));
        }

        return view('customers.import');
    }

    public function import(Request $request)
    {
        // Check permissions
        $user = auth()->user();
        if (!$user->hasAnyRole(['Admin', 'Super Admin'])) {
            abort(403, trans('messages.unauthorized_action'));
        }

        $request->validate([
            'file' => 'required|file|mimes:xlsx,xls|max:10240',
        ]);

        try {
            $file = $request->file('file');
            
            if (!$file->isValid()) {
                return back()->with('error', trans('messages.import_error') . ': ' . 'Invalid file.');
            }

            $import = new CustomersImport();
            
            Excel::import($import, $file);
            
            $imported = $import->getImported();
            $failed = $import->getFailed();
            $errors = $import->getErrors();

            if ($imported === 0 && $failed === 0) {
                return back()->with('error', trans('messages.excel_file_empty'));
            }

            $message = trans('messages.import_completed', [
                'imported' => $imported,
                'failed' => $failed,
            ]);

            if ($failed > 0 && !empty($errors)) {
                return back()->with('warning', $message)->with('import_errors', $errors);
            }

            return back()->with('success', $message);

        } catch (\Maatwebsite\Excel\Validators\ValidationException $e) {
            $failures = $e->failures();
            $errors = [];
            
            foreach ($failures as $failure) {
                $errors[] = [
                    'row' => $failure->row(),
                    'name' => $failure->values()['name'] ?? 'N/A',
                    'error' => implode(', ', $failure->errors()),
                ];
            }
            
            return back()->with('error', trans('messages.import_error'))
                        ->with('import_errors', $errors);
                        
        } catch (\Maatwebsite\Excel\Exceptions\SheetNotFoundException $e) {
            return back()->with('error', trans('messages.import_error') . ': ' . 'Sheet not found in Excel file.');
        } catch (\PhpOffice\PhpSpreadsheet\Reader\Exception $e) {
            return back()->with('error', trans('messages.import_error') . ': ' . 'Error reading Excel file. Please check if the file is valid.');
        } catch (\Exception $e) {
            \Log::error('Customer import error: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            return back()->with('error', trans('messages.import_error') . ': ' . $e->getMessage());
        }
    }

    public function downloadTemplate()
    {
        $data = [
            ['name', 'phone', 'email', 'address', 'current_balance'],
            ['Customer 1', '123456789', 'customer1@example.com', 'Address 1', '0.00'],
            ['Customer 2', '987654321', 'customer2@example.com', 'Address 2', '100.00'],
        ];

        $filename = 'customers_import_template_' . date('Y-m-d') . '.xlsx';
        
        $export = new class($data) implements \Maatwebsite\Excel\Concerns\FromArray {
            protected $data;
            
            public function __construct($data) {
                $this->data = $data;
            }
            
            public function array(): array {
                return $this->data;
            }
        };
        
        return Excel::download($export, $filename);
    }
}

