<?php

namespace App\Http\Controllers;

use App\Models\Customer;
use App\Models\Invoice;
use App\Models\Product;
use App\Models\Supplier;
use App\Models\Warehouse;
use App\Models\Setting;
use App\Models\ProductDefect;
use App\Models\QualityComplaint;
use App\Services\NotificationService;
use App\Services\StockManager;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    public function index(NotificationService $notificationService)
    {
        // Generate notifications
        $notificationService->generateAllNotifications();

        $todaySales = Invoice::whereDate('created_at', today())
            ->whereIn('status', ['paid', 'partially_paid', 'final'])
            ->sum('total');

        $monthSales = Invoice::whereMonth('created_at', now()->month)
            ->whereYear('created_at', now()->year)
            ->whereIn('status', ['paid', 'partially_paid', 'final'])
            ->sum('total');

        $pendingInvoices = Invoice::pending()->count();

        $lowStockProducts = Product::lowStock()->count();

        $totalCustomers = Customer::count();

        $totalSuppliers = Supplier::count();

        // Dashboard settings
        $showRecentInvoices = Setting::get('dashboard_show_recent_invoices', '1') === '1';
        $showLowStock = Setting::get('dashboard_show_low_stock', '1') === '1';
        $recentInvoicesCount = (int)Setting::get('dashboard_recent_invoices_count', 10);
        $lowStockCount = (int)Setting::get('dashboard_low_stock_count', 10);

        // Get recent invoices only if enabled
        $recentInvoices = collect();
        if ($showRecentInvoices) {
            $recentInvoices = Invoice::with(['customer', 'user'])
                ->latest()
                ->take($recentInvoicesCount)
                ->get();
        }

        // Get low stock items only if enabled
        $lowStockItems = collect();
        if ($showLowStock) {
            $lowStockItems = Product::with('category')
                ->lowStock()
                ->take($lowStockCount)
                ->get();
        }

        // Get notifications
        $notifications = $notificationService->getRecentNotifications(10);
        $unreadCount = $notificationService->getUnreadCount();

        // Warehouse statistics (only if multiple warehouses mode is enabled)
        $warehouseStats = null;
        if (StockManager::isMultipleWarehouses()) {
            $warehouses = Warehouse::active()->get();
            $totalWarehouses = $warehouses->count();
            $totalWarehouseStockValue = $warehouses->sum('total_stock_value');
            $totalWarehouseProducts = $warehouses->sum('total_products_count');
            $warehouseStats = [
                'total_warehouses' => $totalWarehouses,
                'total_stock_value' => $totalWarehouseStockValue,
                'total_products' => $totalWarehouseProducts,
                'warehouses' => $warehouses,
            ];
        }

        // Quality Management Statistics (this month)
        $qualityStats = null;
        $showQuality = Setting::get('dashboard_show_quality', '1') === '1';
        if ($showQuality && (auth()->user()->can('quality.reports.view') || 
            auth()->user()->can('quality.defects.view') || 
            auth()->user()->can('quality.complaints.view'))) {
            
            $monthStart = now()->startOfMonth();
            $monthEnd = now()->endOfMonth();
            
            // Defects Statistics
            $totalDefects = ProductDefect::whereBetween('reported_date', [$monthStart, $monthEnd])->count();
            $resolvedDefects = ProductDefect::whereBetween('reported_date', [$monthStart, $monthEnd])
                ->where('status', 'resolved')->count();
            $criticalDefects = ProductDefect::whereBetween('reported_date', [$monthStart, $monthEnd])
                ->where('severity', 'critical')->count();
            
            // Complaints Statistics
            $totalComplaints = QualityComplaint::whereBetween('reported_date', [$monthStart, $monthEnd])->count();
            $resolvedComplaints = QualityComplaint::whereBetween('reported_date', [$monthStart, $monthEnd])
                ->whereIn('status', ['resolved', 'closed'])->count();
            $criticalComplaints = QualityComplaint::whereBetween('reported_date', [$monthStart, $monthEnd])
                ->where('severity', 'critical')->count();
            
            $qualityStats = [
                'total_defects' => $totalDefects,
                'resolved_defects' => $resolvedDefects,
                'critical_defects' => $criticalDefects,
                'total_complaints' => $totalComplaints,
                'resolved_complaints' => $resolvedComplaints,
                'critical_complaints' => $criticalComplaints,
            ];
        }

        return view('dashboard', compact(
            'todaySales',
            'monthSales',
            'pendingInvoices',
            'lowStockProducts',
            'totalCustomers',
            'totalSuppliers',
            'recentInvoices',
            'lowStockItems',
            'notifications',
            'unreadCount',
            'warehouseStats',
            'qualityStats',
            'showRecentInvoices',
            'showLowStock'
        ));
    }
}

