<?php

namespace App\Http\Controllers;

use App\Models\Inventory;
use App\Models\InventoryItem;
use App\Models\Product;
use App\Models\Setting;
use Illuminate\Http\Request;
use App\Traits\Sortable;
use Illuminate\Support\Facades\DB;

class InventoryController extends Controller
{
    use Sortable;

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Inventory::with(['user', 'items']);

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('reference', 'like', "%{$search}%")
                  ->orWhere('notes', 'like', "%{$search}%");
            });
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $itemsPerPage = Setting::getItemsPerPage(20);
        $inventories = $query->latest()->paginate($itemsPerPage)->appends($request->query());

        return view('inventories.index', compact('inventories'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $products = Product::with('category')->orderBy('name')->get();
        $reference = $this->generateReference();
        return view('inventories.create', compact('products', 'reference'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'reference' => 'required|unique:inventories,reference',
            'inventory_date' => 'required|date',
            'notes' => 'nullable|string',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.physical_quantity' => 'required|integer|min:0',
            'items.*.notes' => 'nullable|string',
        ]);

        $inventory = Inventory::create([
            'reference' => $validated['reference'],
            'inventory_date' => $validated['inventory_date'],
            'notes' => $validated['notes'],
            'user_id' => auth()->id(),
            'status' => 'draft',
        ]);

        foreach ($validated['items'] as $item) {
            $product = Product::find($item['product_id']);
            $difference = $item['physical_quantity'] - $product->stock_quantity;

            InventoryItem::create([
                'inventory_id' => $inventory->id,
                'product_id' => $item['product_id'],
                'system_quantity' => $product->stock_quantity,
                'physical_quantity' => $item['physical_quantity'],
                'difference' => $difference,
                'notes' => $item['notes'] ?? null,
            ]);
        }

        return redirect()->route('inventories.show', $inventory)
            ->with('success', trans('messages.inventory_created'));
    }

    /**
     * Display the specified resource.
     */
    public function show(Inventory $inventory)
    {
        $inventory->load(['user', 'items.product.category']);
        return view('inventories.show', compact('inventory'));
    }

    /**
     * Complete the inventory and update stock
     */
    public function complete(Inventory $inventory)
    {
        if ($inventory->status !== 'draft') {
            return back()->with('error', trans('messages.inventory_already_processed'));
        }

        DB::transaction(function () use ($inventory) {
            foreach ($inventory->items as $item) {
                if ($item->difference != 0) {
                    // تحديث كمية المخزون
                    $item->product->stock_quantity = $item->physical_quantity;
                    $item->product->save();

                    // تسجيل حركة المخزون
                    \App\Models\StockMovement::create([
                        'product_id' => $item->product_id,
                        'type' => $item->difference > 0 ? 'IN' : 'OUT',
                        'quantity' => abs($item->difference),
                        'quantity_before' => $item->system_quantity,
                        'quantity_after' => $item->physical_quantity,
                        'reference_type' => Inventory::class,
                        'reference_id' => $inventory->id,
                        'notes' => 'Inventory Adjustment: ' . $inventory->reference,
                        'user_id' => auth()->id(),
                    ]);
                }
            }

            $inventory->status = 'completed';
            $inventory->save();
        });

        return redirect()->route('inventories.show', $inventory)
            ->with('success', trans('messages.inventory_completed'));
    }

    /**
     * Print inventory
     */
    public function print(Inventory $inventory)
    {
        $inventory->load(['user', 'items.product.category']);
        return view('inventories.print', compact('inventory'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Inventory $inventory)
    {
        if ($inventory->status === 'completed') {
            return back()->with('error', trans('messages.cannot_delete_completed_inventory'));
        }

        $inventory->delete();

        return redirect()->route('inventories.index')
            ->with('success', trans('messages.inventory_deleted'));
    }

    /**
     * Generate unique reference for inventory
     */
    private function generateReference()
    {
        $prefix = 'STK-';
        $lastInventory = Inventory::where('reference', 'like', $prefix . '%')
            ->orderByRaw('LENGTH(reference) DESC, reference DESC')
            ->first();

        if ($lastInventory) {
            $numberPart = substr($lastInventory->reference, strlen($prefix));
            if (preg_match('/^(\d+)/', $numberPart, $matches)) {
                $nextNumber = (int)$matches[1] + 1;
            } else {
                $nextNumber = 1;
            }
        } else {
            $nextNumber = 1;
        }

        return $prefix . str_pad($nextNumber, 4, '0', STR_PAD_LEFT);
    }
}
