<?php

namespace App\Http\Controllers;

use App\Models\Customer;
use App\Models\Invoice;
use App\Models\InvoiceItem;
use App\Models\Product;
use App\Models\Setting;
use App\Models\StockMovement;
use App\Services\StockManager;
use App\Models\Treasury;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Traits\Sortable;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\InvoicesExport;
use App\Imports\InvoicesImport;
use App\Mail\InvoiceEmail;
use App\Mail\InvoiceReminder;

class InvoiceController extends Controller
{
    use Sortable;

    public function index(Request $request)
    {
        // Show all invoices (final and draft), but exclude quotations (DEV-)
        $query = Invoice::with(['customer', 'user'])
            ->where('invoice_number', 'not like', 'DEV-%'); // Exclude quotations

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('invoice_number', 'like', "%{$search}%")
                  ->orWhere('customer_name', 'like', "%{$search}%")
                  ->orWhereHas('customer', function ($q) use ($search) {
                      $q->where('name', 'like', "%{$search}%");
                  });
            });
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $allowedColumns = ['invoice_number', 'created_at', 'total', 'paid_amount', 'due_amount', 'status'];
        $sort = $request->get('sort', 'created_at');
        $direction = $request->get('direction', 'desc');

        if (in_array($sort, $allowedColumns)) {
            $query->orderBy($sort, $direction);
        } else {
            $query->latest();
        }

        $itemsPerPage = Setting::getItemsPerPage(20);
        $invoices = $query->paginate($itemsPerPage)->appends($request->query());
        $customers = Customer::all();

        return view('invoices.index', compact('invoices', 'customers', 'sort', 'direction'));
    }

    public function create(Request $request)
    {
        $customers = Customer::all();
        $products = Product::all(); // Allow all products for invoices (not just in stock)
        return view('invoices.create', compact('customers', 'products'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'customer_id' => 'nullable|exists:customers,id',
            'customer_name' => 'nullable|string|max:255',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.tax_rate' => 'nullable|numeric|min:0|max:100',
            'items.*.discount' => 'nullable|numeric|min:0',
            'discount' => 'nullable|numeric|min:0',
            'notes' => 'nullable|string',
        ]);

        DB::beginTransaction();
        try {
            $invoice = Invoice::create([
                'invoice_number' => $this->generateInvoiceNumber(),
                'customer_id' => $validated['customer_id'] ?? null,
                'customer_name' => $validated['customer_name'] ?? 'Walk-in Customer',
                'status' => 'draft', // Create as draft invoice
                'discount' => $validated['discount'] ?? 0,
                'notes' => $validated['notes'] ?? null,
                'user_id' => auth()->id(),
            ]);

            // Create invoice items without deducting stock (stock will be deducted when finalized)
            foreach ($validated['items'] as $itemData) {
                $product = Product::findOrFail($itemData['product_id']);

                $item = InvoiceItem::create([
                    'invoice_id' => $invoice->id,
                    'product_id' => $itemData['product_id'],
                    'quantity' => $itemData['quantity'],
                    'unit_price' => $itemData['unit_price'],
                    'tax_rate' => $itemData['tax_rate'] ?? $product->tax_rate,
                    'discount' => $itemData['discount'] ?? 0,
                    'subtotal' => 0,
                    'total' => 0,
                ]);

                $item->calculateTotal();
            }

            $invoice->updateTotals();

            DB::commit();

            return redirect()->route('invoices.show', $invoice)
                ->with('success', trans('messages.invoice_created'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.error_creating_invoice') . ': ' . $e->getMessage())
                ->withInput();
        }
    }

    public function show(Invoice $invoice)
    {
        $invoice->load(['customer', 'user', 'items.product', 'items.batch.supplier', 'payments.user']);
        return view('invoices.show', compact('invoice'));
    }

    public function edit(Invoice $invoice)
    {
        if (!$invoice->isDraft()) {
            return redirect()->route('invoices.show', $invoice)
                ->with('error', trans('messages.only_draft_can_be_edited'));
        }

        $invoice->load('items.product');
        $customers = Customer::all();
        $products = Product::all();
        
        return view('invoices.edit', compact('invoice', 'customers', 'products'));
    }

    public function update(Request $request, Invoice $invoice)
    {
        if (!$invoice->isDraft()) {
            return back()->with('error', trans('messages.only_draft_can_be_edited'));
        }

        $validated = $request->validate([
            'customer_id' => 'nullable|exists:customers,id',
            'customer_name' => 'nullable|string|max:255',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.tax_rate' => 'nullable|numeric|min:0|max:100',
            'items.*.discount' => 'nullable|numeric|min:0',
            'discount' => 'nullable|numeric|min:0',
            'notes' => 'nullable|string',
        ]);

        DB::beginTransaction();
        try {
            $invoice->update([
                'customer_id' => $validated['customer_id'] ?? null,
                'customer_name' => $validated['customer_name'] ?? 'Walk-in Customer',
                'discount' => $validated['discount'] ?? 0,
                'notes' => $validated['notes'] ?? null,
            ]);

            $invoice->items()->delete();

            foreach ($validated['items'] as $itemData) {
                $product = Product::findOrFail($itemData['product_id']);
                
                $item = InvoiceItem::create([
                    'invoice_id' => $invoice->id,
                    'product_id' => $itemData['product_id'],
                    'quantity' => $itemData['quantity'],
                    'unit_price' => $itemData['unit_price'],
                    'tax_rate' => $itemData['tax_rate'] ?? $product->tax_rate,
                    'discount' => $itemData['discount'] ?? 0,
                    'subtotal' => 0,
                    'total' => 0,
                ]);

                $item->calculateTotal();
            }

            $invoice->updateTotals();

            DB::commit();

            return redirect()->route('invoices.show', $invoice)
                ->with('success', trans('messages.invoice_updated'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.error_updating_invoice') . ': ' . $e->getMessage())
                ->withInput();
        }
    }

    public function finalize(Invoice $invoice)
    {
        if (!$invoice->isDraft()) {
            return back()->with('error', trans('messages.only_draft_can_be_finalized'));
        }

        DB::beginTransaction();
        try {
            $allowNegativeStock = \App\Models\Setting::get('allow_negative_stock', '0') === '1';
            
            foreach ($invoice->items as $item) {
                $product = $item->product;
                
                // Get warehouse ID if multiple warehouses mode
                $warehouseId = null;
                if (StockManager::isMultipleWarehouses()) {
                    $warehouseId = StockManager::getDefaultWarehouse()?->id;
                }
                
                // Check stock availability (only if negative stock is not allowed)
                if (!$allowNegativeStock && !StockManager::checkStockAvailability($product, $item->quantity, $warehouseId)) {
                    DB::rollBack();
                    return back()->with('error', trans('messages.insufficient_stock') . ': ' . $product->name);
                }

                // Update batches using FIFO (First In First Out)
                $remainingQuantity = $item->quantity;
                $batches = \App\Models\ProductBatch::where('product_id', $product->id)
                    ->where('current_quantity', '>', 0)
                    ->orderBy('purchase_date', 'asc')
                    ->orderBy('id', 'asc')
                    ->get();

                $firstBatch = null;
                foreach ($batches as $batch) {
                    if ($remainingQuantity <= 0) break;
                    
                    if ($firstBatch === null) {
                        $firstBatch = $batch; // Store first batch for invoice item
                    }
                    
                    $quantityToDeduct = min($remainingQuantity, $batch->current_quantity);
                    $batch->current_quantity -= $quantityToDeduct;
                    $batch->save();
                    $remainingQuantity -= $quantityToDeduct;
                }

                // Link invoice item to first batch (for tracking)
                if ($firstBatch) {
                    $item->batch_id = $firstBatch->id;
                    $item->save();
                }

                // Update stock using StockManager (after batches to ensure consistency)
                $notes = "Invoice #{$invoice->invoice_number}" . ($firstBatch ? " - Batch: {$firstBatch->batch_number}" : '');
                StockManager::updateStock(
                    $product,
                    $item->quantity,
                    'OUT',
                    $warehouseId,
                    Invoice::class,
                    $invoice->id,
                    $notes
                );
            }

            $invoice->update(['status' => 'final']);
            $invoice->updateTotals();

            DB::commit();

            return redirect()->route('invoices.show', $invoice)
                ->with('success', trans('messages.invoice_finalized'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.error_finalizing_invoice') . ': ' . $e->getMessage());
        }
    }

    public function return(Invoice $invoice)
    {
        if ($invoice->isReturned()) {
            return back()->with('error', trans('messages.invoice_already_returned'));
        }

        if ($invoice->isDraft()) {
            return back()->with('error', trans('messages.draft_cannot_be_returned'));
        }

        DB::beginTransaction();
        try {
            // Get warehouse ID if multiple warehouses mode
            $warehouseId = null;
            if (StockManager::isMultipleWarehouses()) {
                $warehouseId = StockManager::getDefaultWarehouse()?->id;
            }
            
            foreach ($invoice->items as $item) {
                $product = $item->product;
                
                // Update stock using StockManager
                StockManager::updateStock(
                    $product,
                    $item->quantity,
                    'IN',
                    $warehouseId,
                    Invoice::class,
                    $invoice->id,
                    "Return for Invoice #{$invoice->invoice_number}"
                );
            }

            $invoice->update(['status' => 'returned']);

            // Create treasury entry for refund if invoice was paid
            if ($invoice->paid_amount > 0) {
                Treasury::create([
                    'date' => now()->toDateString(),
                    'type' => 'expense',
                    'reference_type' => 'refund',
                    'reference_id' => $invoice->id,
                    'amount' => $invoice->paid_amount,
                    'description' => trans('messages.sales_refund') . ' - ' . trans('messages.invoice') . ' #' . $invoice->invoice_number,
                    'payment_method' => 'cash', // Default to cash for refunds
                    'created_by' => auth()->id(),
                ]);
            }

            DB::commit();

            return redirect()->route('invoices.show', $invoice)
                ->with('success', trans('messages.invoice_returned'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.error_returning_invoice') . ': ' . $e->getMessage());
        }
    }

    public function destroy(Invoice $invoice)
    {
        // Check permissions - Only Admin and Super Admin can delete invoices
        $user = auth()->user();
        if (!$user->hasAnyRole(['Admin', 'Super Admin'])) {
            abort(403, trans('messages.unauthorized_action'));
        }

        DB::beginTransaction();
        try {
            // If invoice was finalized (not draft), restore stock
            if (!$invoice->isDraft() && $invoice->status !== 'returned') {
                // Get warehouse ID if multiple warehouses mode
                $warehouseId = null;
                if (StockManager::isMultipleWarehouses()) {
                    $warehouseId = StockManager::getDefaultWarehouse()?->id;
                }
                
                foreach ($invoice->items as $item) {
                    $product = $item->product;
                    if ($product) {
                        // Update stock using StockManager
                        StockManager::updateStock(
                            $product,
                            $item->quantity,
                            'IN',
                            $warehouseId,
                            Invoice::class,
                            $invoice->id,
                            "Stock restored - Invoice #{$invoice->invoice_number} deleted"
                        );
                    }
                }
            }

            // Delete related payments
            $invoice->payments()->delete();

            // Delete related invoice items
            $invoice->items()->delete();

            // Log deletion in audit_logs
            // Store invoice data before deletion
            $invoiceData = $invoice->toArray();
            $invoiceId = $invoice->id;
            
            \App\Models\AuditLog::create([
                'action' => 'delete',
                'model_type' => Invoice::class,
                'model_id' => $invoiceId,
                'old_values' => $invoiceData, // Boot method will JSON encode
                'new_values' => null,
                'user_id' => auth()->id(),
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
            ]);

            // Delete the invoice
            $invoiceNumber = $invoice->invoice_number;
            $invoice->delete();

            DB::commit();

            return redirect()->route('invoices.index')
                ->with('success', trans('messages.invoice_deleted_successfully'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.error_deleting_invoice') . ': ' . $e->getMessage());
        }
    }

    public function bulkDelete(Request $request)
    {
        // Check permissions - Only Admin and Super Admin can bulk delete
        $user = auth()->user();
        if (!$user->hasAnyRole(['Admin', 'Super Admin'])) {
            abort(403, trans('messages.unauthorized_action'));
        }

        $request->validate([
            'selected_ids' => 'required|array|min:1',
            'selected_ids.*' => 'exists:invoices,id',
        ]);

        DB::beginTransaction();
        try {
            $selectedIds = $request->selected_ids;
            $invoices = Invoice::with('items.product')->whereIn('id', $selectedIds)->get();
            
            foreach ($invoices as $invoice) {
                // If invoice was finalized (not draft), restore stock
                if (!$invoice->isDraft() && $invoice->status !== 'returned') {
                    foreach ($invoice->items as $item) {
                        $product = $item->product;
                        if ($product) {
                            $quantityBefore = $product->stock_quantity;
                            $product->stock_quantity += $item->quantity;
                            $product->save();

                            // Log stock movement
                            StockMovement::create([
                                'product_id' => $product->id,
                                'type' => 'IN',
                                'quantity' => $item->quantity,
                                'quantity_before' => $quantityBefore,
                                'quantity_after' => $product->stock_quantity,
                                'reference_type' => Invoice::class,
                                'reference_id' => $invoice->id,
                                'notes' => "Stock restored - Invoice #{$invoice->invoice_number} bulk deleted",
                                'user_id' => auth()->id(),
                            ]);
                        }
                    }
                }

                // Delete related payments
                $invoice->payments()->delete();

                // Delete related invoice items
                $invoice->items()->delete();

                // Log deletion in audit_logs
                $invoiceData = $invoice->toArray();
                \App\Models\AuditLog::create([
                    'action' => 'bulk_delete',
                    'model_type' => Invoice::class,
                    'model_id' => $invoice->id,
                    'old_values' => $invoiceData,
                    'new_values' => null,
                    'user_id' => auth()->id(),
                    'ip_address' => request()->ip(),
                    'user_agent' => request()->userAgent(),
                ]);
            }

            // Delete all invoices
            Invoice::whereIn('id', $selectedIds)->delete();

            DB::commit();

            return redirect()->route('invoices.index')
                ->with('success', trans('messages.bulk_delete_success', ['count' => count($selectedIds)]));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.bulk_delete_error') . ': ' . $e->getMessage());
        }
    }

    public function pdf(Invoice $invoice)
    {
        $invoice->load(['customer', 'items.product', 'user']);
        // استخدام صفحة الطباعة لجميع اللغات
        return view('invoices.print', compact('invoice'));
    }

    /**
     * طباعة الفاتورة
     */
    public function print(Invoice $invoice)
    {
        $invoice->load(['customer', 'items.product', 'user']);
        return view('invoices.print', compact('invoice'));
    }

    /**
     * تصدير Devis (عرض السعر)
     */
    public function devis(Invoice $invoice)
    {
        $invoice->load(['customer', 'items.product', 'user']);
        // استخدام صفحة الطباعة لجميع اللغات
        return view('invoices.devis-print', compact('invoice'));
    }

    /**
     * تصدير Bon de Livraison (إذن التوصيل)
     */
    public function bonDeLivraison(Invoice $invoice)
    {
        $invoice->load(['customer', 'items.product', 'user']);
        // استخدام صفحة الطباعة لجميع اللغات
        return view('invoices.bon-livraison-print', compact('invoice'));
    }

    private function generateInvoiceNumber(): string
    {
        // Get invoice number format from settings
        $format = \App\Models\Setting::get('invoice_number_format', 'INV-{YEAR}-{NUMBER}');
        
        // Replace {YEAR} with current year
        $year = date('Y');
        $formatWithYear = str_replace('{YEAR}', $year, $format);
        
        // Extract prefix from format (everything before {NUMBER})
        $prefix = str_replace('{NUMBER}', '', $formatWithYear);
        
        // Find all invoices that match the prefix pattern for current year
        $matchingInvoices = Invoice::where('invoice_number', 'like', $prefix . '%')
            ->where('invoice_number', 'not like', 'DEV-%') // Exclude quotations
            ->get();
        
        // Extract numbers from matching invoices
        $numbers = [];
        foreach ($matchingInvoices as $invoice) {
            // Extract numeric part from the end
            $invoiceNum = $invoice->invoice_number;
            // Remove prefix to get the number part
            $numberPart = str_replace($prefix, '', $invoiceNum);
            // Extract only numeric characters
            preg_match('/\d+/', $numberPart, $matches);
            if (!empty($matches)) {
                $numbers[] = (int)$matches[0];
            }
        }
        
        // Get the next number
        $number = empty($numbers) ? 1 : max($numbers) + 1;
        
        // Replace {NUMBER} with the generated number (padded to 6 digits)
        $invoiceNumber = str_replace('{NUMBER}', str_pad($number, 6, '0', STR_PAD_LEFT), $formatWithYear);
        
        // Ensure uniqueness
        $counter = 0;
        while (Invoice::where('invoice_number', $invoiceNumber)->exists() && $counter < 1000) {
            $number++;
            $invoiceNumber = str_replace('{NUMBER}', str_pad($number, 6, '0', STR_PAD_LEFT), $formatWithYear);
            $counter++;
        }
        
        return $invoiceNumber;
    }

    public function export(Request $request)
    {
        $query = Invoice::with(['customer', 'user']);

        // Apply same filters as index
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('invoice_number', 'like', "%{$search}%")
                  ->orWhere('customer_name', 'like', "%{$search}%")
                  ->orWhereHas('customer', function ($q) use ($search) {
                      $q->where('name', 'like', "%{$search}%");
                  });
            });
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $invoices = $query->get();
        $filename = 'invoices_' . date('Y-m-d_His') . '.xlsx';

        return Excel::download(new InvoicesExport($invoices), $filename);
    }

    public function importForm()
    {
        return view('invoices.import');
    }

    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:xlsx,xls|max:10240',
        ]);

        try {
            $import = new InvoicesImport();
            Excel::import($import, $request->file('file'));

            $imported = $import->getImported();
            $failed = $import->getFailed();
            $errors = $import->getErrors();

            if ($imported === 0 && $failed === 0) {
                return back()->with('error', trans('messages.excel_file_empty'));
            }

            $message = trans('messages.import_completed', [
                'imported' => $imported,
                'failed' => $failed,
            ]);

            if ($failed > 0 && !empty($errors)) {
                return back()->with('warning', $message)->with('import_errors', $errors);
            }

            return back()->with('success', $message);

        } catch (\Maatwebsite\Excel\Validators\ValidationException $e) {
            $failures = $e->failures();
            $errors = [];
            
            foreach ($failures as $failure) {
                $errors[] = [
                    'row' => $failure->row(),
                    'name' => $failure->values()['invoice_number'] ?? 'N/A',
                    'error' => implode(', ', $failure->errors()),
                ];
            }
            
            return back()->with('error', trans('messages.import_error'))
                        ->with('import_errors', $errors);
                        
        } catch (\Maatwebsite\Excel\Exceptions\SheetNotFoundException $e) {
            return back()->with('error', trans('messages.import_error') . ': ' . 'Sheet not found in Excel file.');
        } catch (\PhpOffice\PhpSpreadsheet\Reader\Exception $e) {
            return back()->with('error', trans('messages.import_error') . ': ' . 'Error reading Excel file. Please check if the file is valid.');
        } catch (\Exception $e) {
            \Log::error('Invoice import error: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            return back()->with('error', trans('messages.import_error') . ': ' . $e->getMessage());
        }
    }

    public function downloadTemplate()
    {
        $data = [
            ['invoice_number', 'customer_id', 'customer_name', 'status', 'subtotal', 'tax', 'discount', 'total', 'paid_amount', 'due_amount', 'notes'],
            ['FDV-000001', '1', 'Customer Name', 'draft', '100.00', '10.00', '0.00', '110.00', '0.00', '110.00', 'Notes here'],
            ['FDV-000002', '', 'Walk-in Customer', 'final', '200.00', '20.00', '5.00', '215.00', '100.00', '115.00', ''],
        ];

        $filename = 'invoice_import_template_' . date('Y-m-d') . '.xlsx';
        
        $export = new class($data) implements \Maatwebsite\Excel\Concerns\FromArray {
            protected $data;
            
            public function __construct($data) {
                $this->data = $data;
            }
            
            public function array(): array {
                return $this->data;
            }
        };
        
        return Excel::download($export, $filename);
    }

    /**
     * Send invoice via email
     */
    public function sendEmail(Request $request, Invoice $invoice)
    {
        $validated = $request->validate([
            'email' => 'required|email',
            'message' => 'nullable|string|max:1000',
        ]);

        try {
            $invoice->load(['customer', 'items.product', 'payments']);

            \Illuminate\Support\Facades\Mail::to($validated['email'])
                ->send(new InvoiceEmail($invoice, $validated['message'] ?? null));

            return back()->with('success', trans('messages.invoice_sent_successfully'));
        } catch (\Exception $e) {
            \Log::error('Error sending invoice email: ' . $e->getMessage());
            return back()->with('error', trans('messages.error_sending_invoice') . ': ' . $e->getMessage());
        }
    }

    /**
     * Send reminder for invoice
     */
    public function sendReminder(Invoice $invoice)
    {
        if (!$invoice->customer || !$invoice->customer->email) {
            return back()->with('error', trans('messages.customer_email_not_found'));
        }

        if ($invoice->due_amount <= 0) {
            return back()->with('error', trans('messages.invoice_already_paid'));
        }

        try {
            $daysOverdue = \Carbon\Carbon::now()->diffInDays($invoice->created_at);
            
            \Illuminate\Support\Facades\Mail::to($invoice->customer->email)
                ->send(new InvoiceReminder($invoice, $daysOverdue));

            return back()->with('success', trans('messages.reminder_sent_successfully'));
        } catch (\Exception $e) {
            \Log::error('Error sending invoice reminder: ' . $e->getMessage());
            return back()->with('error', trans('messages.error_sending_reminder') . ': ' . $e->getMessage());
        }
    }

    /**
     * Convert quotation (devis) to sales invoice
     */
    public function convertQuotationToInvoice(Invoice $invoice)
    {
        if ($invoice->status !== 'draft') {
            return back()->with('error', trans('messages.can_only_convert_draft_invoice'));
        }

        // Same logic as finalize but with different success message
        if (!$invoice->isDraft()) {
            return back()->with('error', trans('messages.only_draft_can_be_finalized'));
        }

        DB::beginTransaction();
        try {
            // Save the original quotation number before changing it
            $originalQuotationNumber = $invoice->invoice_number;
            
            // Generate new invoice number (not quotation number)
            $newInvoiceNumber = $this->generateInvoiceNumber();
            
            // Update invoice with new number and status (keep as draft, no stock deduction)
            // Stock will be deducted when invoice is finalized
            $invoice->update([
                'status' => 'draft',
                'invoice_number' => $newInvoiceNumber,
                'quotation_number' => $originalQuotationNumber // Save original quotation number
            ]);
            $invoice->updateTotals();

            DB::commit();

            return redirect()->route('invoices.show', $invoice)
                ->with('success', trans('messages.quotation_converted_to_invoice'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.error_occurred') . ': ' . $e->getMessage());
        }
    }

    /**
     * Display quotations (draft invoices)
     */
    public function quotations(Request $request)
    {
        // Show only quotations (draft invoices with DEV- prefix)
        $query = Invoice::with(['customer', 'user'])
            ->where('status', 'draft')
            ->where('invoice_number', 'like', 'DEV-%');

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('invoice_number', 'like', "%{$search}%")
                  ->orWhere('customer_name', 'like', "%{$search}%")
                  ->orWhereHas('customer', function ($q) use ($search) {
                      $q->where('name', 'like', "%{$search}%");
                  });
            });
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $allowedColumns = ['invoice_number', 'created_at', 'total'];
        $sort = $request->get('sort', 'created_at');
        $direction = $request->get('direction', 'desc');

        if (in_array($sort, $allowedColumns)) {
            $query->orderBy($sort, $direction);
        } else {
            $query->latest();
        }

        $itemsPerPage = Setting::getItemsPerPage(20);
        $quotations = $query->paginate($itemsPerPage)->appends($request->query());
        $customers = Customer::all();

        return view('quotations.index', compact('quotations', 'customers', 'sort', 'direction'));
    }

    /**
     * Create quotation
     */
    public function createQuotation()
    {
        $customers = Customer::all();
        $products = Product::all(); // Allow all products for quotations
        return view('quotations.create', compact('customers', 'products'));
    }

    /**
     * Store quotation
     */
    public function storeQuotation(Request $request)
    {
        $validated = $request->validate([
            'customer_id' => 'nullable|exists:customers,id',
            'customer_name' => 'nullable|string|max:255',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.tax_rate' => 'nullable|numeric|min:0|max:100',
            'items.*.discount' => 'nullable|numeric|min:0',
            'discount' => 'nullable|numeric|min:0',
            'notes' => 'nullable|string',
        ]);

        DB::beginTransaction();
        try {
            $quotation = Invoice::create([
                'invoice_number' => $this->generateQuotationNumber(),
                'customer_id' => $validated['customer_id'] ?? null,
                'customer_name' => $validated['customer_name'] ?? 'Walk-in Customer',
                'status' => 'draft', // Quotation is always draft
                'discount' => $validated['discount'] ?? 0,
                'notes' => $validated['notes'] ?? null,
                'user_id' => auth()->id(),
            ]);

            foreach ($validated['items'] as $itemData) {
                $product = Product::findOrFail($itemData['product_id']);
                
                $item = InvoiceItem::create([
                    'invoice_id' => $quotation->id,
                    'product_id' => $itemData['product_id'],
                    'quantity' => $itemData['quantity'],
                    'unit_price' => $itemData['unit_price'],
                    'tax_rate' => $itemData['tax_rate'] ?? $product->tax_rate,
                    'discount' => $itemData['discount'] ?? 0,
                    'subtotal' => 0,
                    'total' => 0,
                ]);

                $item->calculateTotal();
            }

            $quotation->updateTotals();

            DB::commit();

            return redirect()->route('quotations.show', $quotation)
                ->with('success', trans('messages.quotation_created'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.error_creating_quotation') . ': ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Show quotation
     */
    public function showQuotation(Invoice $invoice)
    {
        if ($invoice->status !== 'draft') {
            return redirect()->route('invoices.show', $invoice);
        }
        $invoice->load(['customer', 'user', 'items.product', 'items.batch.supplier']);
        return view('quotations.show', compact('invoice'));
    }

    /**
     * Edit quotation
     */
    public function editQuotation(Invoice $invoice)
    {
        if ($invoice->status !== 'draft') {
            return redirect()->route('quotations.show', $invoice)
                ->with('error', trans('messages.only_draft_can_be_edited'));
        }

        $invoice->load('items.product');
        $customers = Customer::all();
        $products = Product::all();
        
        return view('quotations.edit', compact('invoice', 'customers', 'products'));
    }

    /**
     * Update quotation
     */
    public function updateQuotation(Request $request, Invoice $invoice)
    {
        if ($invoice->status !== 'draft') {
            return back()->with('error', trans('messages.only_draft_can_be_edited'));
        }

        $validated = $request->validate([
            'customer_id' => 'nullable|exists:customers,id',
            'customer_name' => 'nullable|string|max:255',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.tax_rate' => 'nullable|numeric|min:0|max:100',
            'items.*.discount' => 'nullable|numeric|min:0',
            'discount' => 'nullable|numeric|min:0',
            'notes' => 'nullable|string',
        ]);

        DB::beginTransaction();
        try {
            $invoice->update([
                'customer_id' => $validated['customer_id'] ?? null,
                'customer_name' => $validated['customer_name'] ?? 'Walk-in Customer',
                'discount' => $validated['discount'] ?? 0,
                'notes' => $validated['notes'] ?? null,
            ]);

            $invoice->items()->delete();

            foreach ($validated['items'] as $itemData) {
                $product = Product::findOrFail($itemData['product_id']);
                
                $item = InvoiceItem::create([
                    'invoice_id' => $invoice->id,
                    'product_id' => $itemData['product_id'],
                    'quantity' => $itemData['quantity'],
                    'unit_price' => $itemData['unit_price'],
                    'tax_rate' => $itemData['tax_rate'] ?? $product->tax_rate,
                    'discount' => $itemData['discount'] ?? 0,
                    'subtotal' => 0,
                    'total' => 0,
                ]);

                $item->calculateTotal();
            }

            $invoice->updateTotals();

            DB::commit();

            return redirect()->route('quotations.show', $invoice)
                ->with('success', trans('messages.quotation_updated'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.error_updating_quotation') . ': ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Destroy quotation
     */
    public function destroyQuotation(Invoice $invoice)
    {
        if ($invoice->status !== 'draft') {
            return back()->with('error', trans('messages.can_only_delete_draft_quotation'));
        }

        DB::beginTransaction();
        try {
            $invoice->items()->delete();
            $invoice->delete();

            DB::commit();

            return redirect()->route('quotations.index')
                ->with('success', trans('messages.quotation_deleted_successfully'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.error_deleting_quotation') . ': ' . $e->getMessage());
        }
    }

    /**
     * Generate quotation number
     */
    private function generateQuotationNumber(): string
    {
        // Get last quotation (regardless of status) that starts with DEV-
        $lastQuotation = Invoice::where('invoice_number', 'like', 'DEV-%')
            ->latest()
            ->first();
        $number = $lastQuotation ? (int) substr($lastQuotation->invoice_number, -6) + 1 : 1;
        return 'DEV-' . str_pad($number, 6, '0', STR_PAD_LEFT);
    }

    /**
     * Bulk delete quotations
     */
    public function bulkDeleteQuotations(Request $request)
    {
        $request->validate([
            'selected_ids' => 'required|array|min:1',
            'selected_ids.*' => 'exists:invoices,id',
        ]);

        DB::beginTransaction();
        try {
            $selectedIds = $request->selected_ids;
            $quotations = Invoice::whereIn('id', $selectedIds)->where('status', 'draft')->get();
            
            $deletedCount = 0;
            foreach ($quotations as $quotation) {
                $quotation->items()->delete();
                $quotation->delete();
                $deletedCount++;
            }

            DB::commit();
            return redirect()->route('quotations.index')
                ->with('success', trans('messages.bulk_delete_success', ['count' => $deletedCount]));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.bulk_delete_error') . ': ' . $e->getMessage());
        }
    }
}

