<?php

namespace App\Http\Controllers;

use App\Models\Category;
use App\Models\Product;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;
use App\Traits\Sortable;
use App\Traits\ChecksPermissions;
use Maatwebsite\Excel\Facades\Excel;
use App\Imports\ProductsImport;

class ProductController extends Controller
{
    use Sortable, ChecksPermissions;

    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index(Request $request)
    {
        $this->checkPermission('products.view');
        $query = Product::with('category');

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('name_ar', 'like', "%{$search}%")
                  ->orWhere('sku', 'like', "%{$search}%")
                  ->orWhere('barcode', 'like', "%{$search}%");
            });
        }

        if ($request->filled('category_id')) {
            $query->where('category_id', $request->category_id);
        }

        if ($request->filled('low_stock')) {
            $query->lowStock();
        }

        $allowedColumns = ['sku', 'name', 'stock_quantity', 'purchase_price', 'selling_price', 'created_at'];
        $sort = $request->get('sort', 'created_at');
        $direction = $request->get('direction', 'desc');

        if (in_array($sort, $allowedColumns)) {
            if ($sort === 'name') {
                $query->orderBy('name', $direction);
            } else {
                $query->orderBy($sort, $direction);
            }
        } else {
            $query->latest();
        }

        $itemsPerPage = Setting::getItemsPerPage(20);
        $products = $query->paginate($itemsPerPage)->appends($request->query());
        $categories = Category::all();

        return view('products.index', compact('products', 'categories', 'sort', 'direction'));
    }

    public function create()
    {
        $this->checkPermission('products.create');
        $categories = Category::all();
        $skuGenerationMode = \App\Models\Setting::get('sku_generation_mode', 'auto');
        $skuPrefix = \App\Models\Setting::get('sku_prefix', 'PROD-');
        return view('products.create', compact('categories', 'skuGenerationMode', 'skuPrefix'));
    }

    public function store(Request $request)
    {
        $this->checkPermission('products.create');
        $skuGenerationMode = \App\Models\Setting::get('sku_generation_mode', 'auto');
        
        // Generate SKU automatically if mode is auto
        if ($skuGenerationMode === 'auto') {
            $skuPrefix = \App\Models\Setting::get('sku_prefix', 'PROD-');
            $request->merge(['sku' => $this->generateSku($skuPrefix)]);
        }
        
        $validated = $request->validate([
            'sku' => 'required|unique:products,sku',
            'barcode' => 'nullable|unique:products,barcode',
            'name' => 'required|string|max:255',
            'name_ar' => 'nullable|string|max:255',
            'brand' => 'nullable|string|max:255',
            'category_id' => 'nullable|exists:categories,id',
            'unit' => 'required|string|max:50',
            'purchase_price' => 'required|numeric|min:0',
            'selling_price' => 'required|numeric|min:0',
            'tax_rate' => 'nullable|numeric|min:0|max:100',
            'stock_quantity' => 'required|integer|min:0',
            'reorder_level' => 'required|integer|min:0',
            'image' => 'nullable|image|max:2048',
            'description' => 'nullable|string',
        ]);

        if ($request->hasFile('image')) {
            $validated['image'] = $request->file('image')->store('products', 'public');
        }

        Product::create($validated);

        return redirect()->route('products.index')
            ->with('success', trans('messages.product_created'));
    }

    public function show(Product $product)
    {
        $this->checkPermission('products.view');
        $product->load([
            'category', 
            'stockMovements.user', 
            'invoiceItems.invoice',
            'batches.supplier',
            'batches.purchaseInvoice',
            'warehouses'
        ]);
        return view('products.show', compact('product'));
    }

    public function edit(Product $product)
    {
        $this->checkPermission('products.update');
        $categories = Category::all();
        $skuGenerationMode = \App\Models\Setting::get('sku_generation_mode', 'auto');
        $skuPrefix = \App\Models\Setting::get('sku_prefix', 'PROD-');
        return view('products.edit', compact('product', 'categories', 'skuGenerationMode', 'skuPrefix'));
    }

    public function update(Request $request, Product $product)
    {
        $this->checkPermission('products.update');
        $validated = $request->validate([
            'sku' => ['required', Rule::unique('products')->ignore($product->id)],
            'barcode' => ['nullable', Rule::unique('products')->ignore($product->id)],
            'name' => 'required|string|max:255',
            'name_ar' => 'nullable|string|max:255',
            'brand' => 'nullable|string|max:255',
            'category_id' => 'nullable|exists:categories,id',
            'unit' => 'required|string|max:50',
            'purchase_price' => 'required|numeric|min:0',
            'selling_price' => 'required|numeric|min:0',
            'tax_rate' => 'nullable|numeric|min:0|max:100',
            'stock_quantity' => 'required|integer|min:0',
            'reorder_level' => 'required|integer|min:0',
            'image' => 'nullable|image|max:2048',
            'description' => 'nullable|string',
        ]);

        if ($request->hasFile('image')) {
            if ($product->image) {
                Storage::disk('public')->delete($product->image);
            }
            $validated['image'] = $request->file('image')->store('products', 'public');
        }

        $product->update($validated);

        return redirect()->route('products.index')
            ->with('success', trans('messages.product_updated'));
    }

    public function destroy(Product $product)
    {
        $this->checkPermission('products.delete');
        if ($product->image) {
            Storage::disk('public')->delete($product->image);
        }

        $product->delete();

        return redirect()->route('products.index')
            ->with('success', trans('messages.product_deleted'));
    }

    public function export()
    {
        $this->checkPermission('products.export');
        // Excel export implementation
        $filename = 'products_' . date('Y-m-d') . '.xlsx';
        return Excel::download(new \App\Exports\ProductsExport(), $filename);
    }

    public function importForm()
    {
        $this->checkPermission('products.import');

        return view('products.import');
    }

    public function import(Request $request)
    {
        $this->checkPermission('products.import');

        $request->validate([
            'file' => 'required|mimes:xlsx,xls|max:10240', // 10MB max
        ]);

        try {
            // Check if file was uploaded
            if (!$request->hasFile('file')) {
                return back()->with('error', trans('messages.import_error') . ': ' . 'No file uploaded.');
            }

            $file = $request->file('file');
            
            // Check if file is valid
            if (!$file->isValid()) {
                return back()->with('error', trans('messages.import_error') . ': ' . 'Invalid file.');
            }

            $import = new ProductsImport();
            
            Excel::import($import, $file);
            
            $imported = $import->getImported();
            $failed = $import->getFailed();
            $errors = $import->getErrors();

            if ($imported === 0 && $failed === 0) {
                return back()->with('error', trans('messages.excel_file_empty'));
            }

            $message = trans('messages.import_completed', [
                'imported' => $imported,
                'failed' => $failed,
            ]);

            if ($failed > 0 && !empty($errors)) {
                return back()->with('warning', $message)->with('import_errors', $errors);
            }

            return back()->with('success', $message);

        } catch (\Maatwebsite\Excel\Validators\ValidationException $e) {
            $failures = $e->failures();
            $errors = [];
            
            foreach ($failures as $failure) {
                $errors[] = [
                    'row' => $failure->row(),
                    'name' => $failure->values()['name'] ?? 'N/A',
                    'error' => implode(', ', $failure->errors()),
                ];
            }
            
            return back()->with('error', trans('messages.import_error'))
                        ->with('import_errors', $errors);
                        
        } catch (\Maatwebsite\Excel\Exceptions\SheetNotFoundException $e) {
            return back()->with('error', trans('messages.import_error') . ': ' . 'Sheet not found in Excel file.');
        } catch (\PhpOffice\PhpSpreadsheet\Reader\Exception $e) {
            return back()->with('error', trans('messages.import_error') . ': ' . 'Error reading Excel file. Please check if the file is valid.');
        } catch (\Exception $e) {
            \Log::error('Product import error: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            return back()->with('error', trans('messages.import_error') . ': ' . $e->getMessage());
        }
    }

    public function downloadTemplate()
    {
        $data = [
            ['name', 'barcode', 'sku', 'purchase_price', 'Price', 'qt_stock', 'brand', 'description'],
            ['Product Name 1', '123456789', 'SKU001', '12.00', '15.00', '100', 'Brand Name 1', 'Product description'],
            ['Product Name 2', '987654321', 'SKU002', '25.00', '30.00', '50', 'Brand Name 2', 'Another product'],
        ];

        $filename = 'product_import_template_' . date('Y-m-d') . '.xlsx';
        
        $export = new class($data) implements \Maatwebsite\Excel\Concerns\FromArray {
            protected $data;
            
            public function __construct($data) {
                $this->data = $data;
            }
            
            public function array(): array {
                return $this->data;
            }
        };
        
        return Excel::download($export, $filename);
    }

    public function bulkDelete(Request $request)
    {
        $this->checkPermission('products.bulk_delete');

        $request->validate([
            'selected_ids' => 'required|array|min:1',
            'selected_ids.*' => 'exists:products,id',
        ]);

        try {
            $selectedIds = $request->selected_ids;
            $products = Product::whereIn('id', $selectedIds)->get();
            
            // Log before deletion
            foreach ($products as $product) {
                \App\Models\AuditLog::create([
                    'action' => 'bulk_delete',
                    'model_type' => Product::class,
                    'model_id' => $product->id,
                    'old_values' => $product->toArray(),
                    'new_values' => null,
                    'user_id' => auth()->id(),
                    'ip_address' => request()->ip(),
                    'user_agent' => request()->userAgent(),
                ]);
            }

            Product::whereIn('id', $selectedIds)->delete();

            return redirect()->route('products.index')
                ->with('success', trans('messages.bulk_delete_success', ['count' => count($selectedIds)]));
        } catch (\Exception $e) {
            return back()->with('error', trans('messages.bulk_delete_error') . ': ' . $e->getMessage());
        }
    }

    public function printLowStock()
    {
        // Get low stock products: stock_quantity <= reorder_level
        $products = Product::with('category')
            ->lowStock()
            ->orderBy('stock_quantity', 'asc')
            ->orderBy('name', 'asc')
            ->get();

        // استخدام صفحة الطباعة لجميع اللغات
        return view('products.print.low-stock', compact('products'));
    }

    public function checkSku(Request $request)
    {
        $sku = $request->input('sku');
        $productId = $request->input('product_id'); // For edit mode
        
        if (empty($sku)) {
            return response()->json([
                'available' => false,
                'message' => trans('messages.sku_required')
            ]);
        }

        $query = Product::where('sku', $sku);
        
        // Exclude current product when editing
        if ($productId) {
            $query->where('id', '!=', $productId);
        }

        $exists = $query->exists();

        return response()->json([
            'available' => !$exists,
            'message' => $exists ? trans('messages.sku_already_exists') : trans('messages.sku_available')
        ]);
    }

    private function generateSku($prefix = 'PROD-')
    {
        // Find the last SKU with this prefix
        $lastProduct = Product::where('sku', 'like', $prefix . '%')
            ->orderByRaw('LENGTH(sku) DESC, sku DESC')
            ->first();

        if ($lastProduct) {
            // Extract number from last SKU (remove prefix)
            $numberPart = substr($lastProduct->sku, strlen($prefix));
            // Try to extract numeric part
            if (preg_match('/^(\d+)/', $numberPart, $matches)) {
                $lastNumber = (int)$matches[1];
                $nextNumber = $lastNumber + 1;
            } else {
                $nextNumber = 1;
            }
        } else {
            $nextNumber = 1;
        }

        // Keep trying until we find an available SKU
        $maxAttempts = 1000;
        $attempt = 0;
        do {
            $sku = $prefix . str_pad($nextNumber, 4, '0', STR_PAD_LEFT);
            $exists = Product::where('sku', $sku)->exists();
            if (!$exists) {
                return $sku;
            }
            $nextNumber++;
            $attempt++;
        } while ($attempt < $maxAttempts);

        // Fallback: use timestamp if all attempts fail
        return $prefix . time();
    }

    public function reports(Request $request)
    {
        $this->checkPermission('products.reports');
        $reportType = $request->get('type', 'best_selling');
        $dateFrom = $request->get('date_from');
        $dateTo = $request->get('date_to');
        $limit = $request->get('limit', 20);

        // Base query for invoice items
        $invoiceItemsQuery = \App\Models\InvoiceItem::with(['product.category', 'invoice'])
            ->whereHas('invoice', function ($q) {
                $q->whereIn('status', ['paid', 'partially_paid', 'final']);
            });

        // Apply date filters
        if ($dateFrom) {
            $invoiceItemsQuery->whereHas('invoice', function ($q) use ($dateFrom) {
                $q->whereDate('created_at', '>=', $dateFrom);
            });
        }

        if ($dateTo) {
            $invoiceItemsQuery->whereHas('invoice', function ($q) use ($dateTo) {
                $q->whereDate('created_at', '<=', $dateTo);
            });
        }

        $invoiceItems = $invoiceItemsQuery->get();

        // Group by product and calculate statistics
        $productStats = $invoiceItems->groupBy('product_id')->map(function ($group) {
            $product = $group->first()->product;
            if (!$product) return null;

            $quantitySold = $group->sum('quantity');
            $totalRevenue = $group->sum('total');
            $totalCost = $group->sum(function ($item) {
                return $item->quantity * ($item->product->purchase_price ?? 0);
            });
            $profit = $totalRevenue - $totalCost;
            $profitMargin = $totalRevenue > 0 ? ($profit / $totalRevenue) * 100 : 0;
            $averagePrice = $quantitySold > 0 ? $totalRevenue / $quantitySold : 0;

            return [
                'product' => $product,
                'quantity_sold' => $quantitySold,
                'total_revenue' => $totalRevenue,
                'total_cost' => $totalCost,
                'profit' => $profit,
                'profit_margin' => $profitMargin,
                'average_price' => $averagePrice,
                'sales_count' => $group->count(),
            ];
        })->filter();

        // Get all products for unsold products report
        $allProducts = Product::with('category')->get();

        // Apply report type filter
        $results = collect();
        switch ($reportType) {
            case 'best_selling':
                $results = $productStats->sortByDesc('quantity_sold')->take($limit);
                break;
            case 'worst_selling':
                $results = $productStats->sortBy('quantity_sold')->take($limit);
                break;
            case 'most_profitable':
                $results = $productStats->sortByDesc('profit')->take($limit);
                break;
            case 'least_profitable':
                $results = $productStats->sortBy('profit')->take($limit);
                break;
            case 'highest_margin':
                $results = $productStats->sortByDesc('profit_margin')->take($limit);
                break;
            case 'lowest_margin':
                $results = $productStats->sortBy('profit_margin')->take($limit);
                break;
            case 'unsold':
                $soldProductIds = $productStats->keys()->toArray();
                $results = $allProducts->whereNotIn('id', $soldProductIds)->take($limit);
                break;
            case 'most_active':
                $results = $productStats->sortByDesc('sales_count')->take($limit);
                break;
            case 'highest_revenue':
                $results = $productStats->sortByDesc('total_revenue')->take($limit);
                break;
            default:
                $results = $productStats->sortByDesc('quantity_sold')->take($limit);
        }

        // Calculate summary statistics
        $summary = [
            'total_products' => $allProducts->count(),
            'sold_products' => $productStats->count(),
            'unsold_products' => $allProducts->count() - $productStats->count(),
            'total_quantity_sold' => $productStats->sum('quantity_sold'),
            'total_revenue' => $productStats->sum('total_revenue'),
            'total_cost' => $productStats->sum('total_cost'),
            'total_profit' => $productStats->sum('profit'),
            'average_profit_margin' => $productStats->count() > 0 ? $productStats->avg('profit_margin') : 0,
        ];

        return view('products.reports', compact('results', 'reportType', 'dateFrom', 'dateTo', 'limit', 'summary'));
    }

}

