<?php

namespace App\Http\Controllers;

use App\Models\PromissoryNote;
use App\Models\Customer;
use App\Models\Supplier;
use App\Models\Invoice;
use App\Models\PurchaseInvoice;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class PromissoryNoteController extends Controller
{
    public function index(Request $request)
    {
        $query = PromissoryNote::with(['customer', 'supplier', 'invoice', 'purchaseInvoice', 'user']);

        // Filter by type
        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by customer
        if ($request->filled('customer_id')) {
            $query->where('customer_id', $request->customer_id);
        }

        // Filter by supplier
        if ($request->filled('supplier_id')) {
            $query->where('supplier_id', $request->supplier_id);
        }

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('note_number', 'like', "%{$search}%")
                  ->orWhereHas('customer', function ($q) use ($search) {
                      $q->where('name', 'like', "%{$search}%");
                  })
                  ->orWhereHas('supplier', function ($q) use ($search) {
                      $q->where('name', 'like', "%{$search}%");
                  });
            });
        }

        // Filter by due date
        if ($request->filled('due_date_from')) {
            $query->where('due_date', '>=', $request->due_date_from);
        }
        if ($request->filled('due_date_to')) {
            $query->where('due_date', '<=', $request->due_date_to);
        }

        $query->orderBy('due_date', 'asc');

        $itemsPerPage = Setting::getItemsPerPage(20);
        $promissoryNotes = $query->paginate($itemsPerPage)->appends($request->query());

        // Update status to 'due' for promissory notes that are past due date
        $promissoryNotes->getCollection()->each(function ($note) {
            if ($note->status === 'pending' && $note->due_date < now() && !$note->isPaid()) {
                $note->update(['status' => 'due']);
            }
        });

        $customers = Customer::orderBy('name')->get();
        $suppliers = Supplier::orderBy('name')->get();

        return view('promissory-notes.index', compact('promissoryNotes', 'customers', 'suppliers'));
    }

    public function create(Request $request)
    {
        $customers = Customer::orderBy('name')->get();
        $suppliers = Supplier::orderBy('name')->get();
        
        $type = $request->get('type', 'customer');
        $customerId = $request->get('customer_id');
        $supplierId = $request->get('supplier_id');
        $invoiceId = $request->get('invoice_id');
        $purchaseInvoiceId = $request->get('purchase_invoice_id');
        $amount = $request->get('amount');

        $invoices = [];
        $purchaseInvoices = [];

        if ($type === 'customer' && $customerId) {
            $query = Invoice::where('customer_id', $customerId)
                ->whereIn('status', ['final', 'partially_paid', 'paid']);
            
            // If a specific invoice_id is provided, include it even if it's paid
            if ($invoiceId) {
                $query->orWhere('id', $invoiceId);
            }
            
            $invoices = $query->orderBy('created_at', 'desc')->get();
        }

        if ($type === 'supplier' && $supplierId) {
            $query = PurchaseInvoice::where('supplier_id', $supplierId)
                ->whereIn('status', ['confirmed', 'partially_paid', 'paid']);
            
            // If a specific purchase_invoice_id is provided, include it even if it's paid
            if ($purchaseInvoiceId) {
                $query->orWhere('id', $purchaseInvoiceId);
            }
            
            $purchaseInvoices = $query->orderBy('created_at', 'desc')->get();
        }

        return view('promissory-notes.create', compact('customers', 'suppliers', 'type', 'customerId', 'supplierId', 'invoiceId', 'purchaseInvoiceId', 'amount', 'invoices', 'purchaseInvoices'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'type' => 'required|in:customer,supplier',
            'customer_id' => 'required_if:type,customer|nullable|exists:customers,id',
            'supplier_id' => 'required_if:type,supplier|nullable|exists:suppliers,id',
            'invoice_id' => 'nullable|exists:invoices,id',
            'purchase_invoice_id' => 'nullable|exists:purchase_invoices,id',
            'amount' => 'required|numeric|min:0.01',
            'note_number' => 'required|string|max:255|unique:promissory_notes,note_number',
            'issue_date' => 'required|date',
            'due_date' => 'required|date|after_or_equal:issue_date',
            'notes' => 'nullable|string',
        ]);

        if ($validated['type'] === 'customer' && !$validated['customer_id']) {
            return back()->withErrors(['customer_id' => trans('messages.customer_required')])->withInput();
        }
        if ($validated['type'] === 'supplier' && !$validated['supplier_id']) {
            return back()->withErrors(['supplier_id' => trans('messages.supplier_required')])->withInput();
        }

        DB::beginTransaction();
        try {
            $promissoryNote = PromissoryNote::create([
                'type' => $validated['type'],
                'customer_id' => $validated['customer_id'] ?? null,
                'supplier_id' => $validated['supplier_id'] ?? null,
                'invoice_id' => $validated['invoice_id'] ?? null,
                'purchase_invoice_id' => $validated['purchase_invoice_id'] ?? null,
                'amount' => $validated['amount'],
                'note_number' => $validated['note_number'],
                'issue_date' => $validated['issue_date'],
                'due_date' => $validated['due_date'],
                'status' => 'pending',
                'notes' => $validated['notes'] ?? null,
                'user_id' => auth()->id(),
            ]);

            DB::commit();

            return redirect()->route('promissory-notes.index')
                ->with('success', trans('messages.promissory_note_created_successfully'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.error_creating_promissory_note') . ': ' . $e->getMessage())->withInput();
        }
    }

    public function show(PromissoryNote $promissoryNote)
    {
        $promissoryNote->load(['customer', 'supplier', 'invoice', 'purchaseInvoice', 'user']);
        return view('promissory-notes.show', compact('promissoryNote'));
    }

    public function edit(PromissoryNote $promissoryNote)
    {
        $customers = Customer::orderBy('name')->get();
        $suppliers = Supplier::orderBy('name')->get();
        
        $invoices = [];
        $purchaseInvoices = [];

        if ($promissoryNote->type === 'customer' && $promissoryNote->customer_id) {
            $invoices = Invoice::where('customer_id', $promissoryNote->customer_id)
                ->whereIn('status', ['final', 'partially_paid'])
                ->orderBy('created_at', 'desc')
                ->get();
        }

        if ($promissoryNote->type === 'supplier' && $promissoryNote->supplier_id) {
            $purchaseInvoices = PurchaseInvoice::where('supplier_id', $promissoryNote->supplier_id)
                ->whereIn('status', ['confirmed', 'partially_paid'])
                ->orderBy('created_at', 'desc')
                ->get();
        }

        return view('promissory-notes.edit', compact('promissoryNote', 'customers', 'suppliers', 'invoices', 'purchaseInvoices'));
    }

    public function update(Request $request, PromissoryNote $promissoryNote)
    {
        $validated = $request->validate([
            'type' => 'required|in:customer,supplier',
            'customer_id' => 'required_if:type,customer|nullable|exists:customers,id',
            'supplier_id' => 'required_if:type,supplier|nullable|exists:suppliers,id',
            'invoice_id' => 'nullable|exists:invoices,id',
            'purchase_invoice_id' => 'nullable|exists:purchase_invoices,id',
            'amount' => 'required|numeric|min:0.01',
            'note_number' => 'required|string|max:255|unique:promissory_notes,note_number,' . $promissoryNote->id,
            'issue_date' => 'required|date',
            'due_date' => 'required|date|after_or_equal:issue_date',
            'notes' => 'nullable|string',
        ]);

        if ($validated['type'] === 'customer' && !$validated['customer_id']) {
            return back()->withErrors(['customer_id' => trans('messages.customer_required')])->withInput();
        }
        if ($validated['type'] === 'supplier' && !$validated['supplier_id']) {
            return back()->withErrors(['supplier_id' => trans('messages.supplier_required')])->withInput();
        }

        if ($promissoryNote->isPaid()) {
            return back()->with('error', trans('messages.cannot_edit_paid_promissory_note'));
        }

        DB::beginTransaction();
        try {
            $promissoryNote->update([
                'type' => $validated['type'],
                'customer_id' => $validated['customer_id'] ?? null,
                'supplier_id' => $validated['supplier_id'] ?? null,
                'invoice_id' => $validated['invoice_id'] ?? null,
                'purchase_invoice_id' => $validated['purchase_invoice_id'] ?? null,
                'amount' => $validated['amount'],
                'note_number' => $validated['note_number'],
                'issue_date' => $validated['issue_date'],
                'due_date' => $validated['due_date'],
                'notes' => $validated['notes'] ?? null,
            ]);

            DB::commit();

            return redirect()->route('promissory-notes.index')
                ->with('success', trans('messages.promissory_note_updated_successfully'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.error_updating_promissory_note') . ': ' . $e->getMessage())->withInput();
        }
    }

    public function destroy(PromissoryNote $promissoryNote)
    {
        if ($promissoryNote->isPaid()) {
            return back()->with('error', trans('messages.cannot_delete_paid_promissory_note'));
        }

        $promissoryNote->delete();

        return redirect()->route('promissory-notes.index')
            ->with('success', trans('messages.promissory_note_deleted_successfully'));
    }

    public function markAsPaid(Request $request, PromissoryNote $promissoryNote)
    {
        if ($promissoryNote->isPaid()) {
            return back()->with('error', trans('messages.promissory_note_already_paid'));
        }

        $validated = $request->validate([
            'paid_date' => 'required|date|after_or_equal:' . $promissoryNote->issue_date->format('Y-m-d'),
        ]);

        $promissoryNote->update([
            'status' => 'paid',
            'paid_date' => $validated['paid_date'],
        ]);

        return back()->with('success', trans('messages.promissory_note_marked_as_paid'));
    }

    public function cancel(PromissoryNote $promissoryNote)
    {
        if ($promissoryNote->isPaid()) {
            return back()->with('error', trans('messages.cannot_cancel_paid_promissory_note'));
        }

        $promissoryNote->update([
            'status' => 'cancelled',
        ]);

        return back()->with('success', trans('messages.promissory_note_cancelled'));
    }

    public function getInvoices($customerId, Request $request)
    {
        $invoiceId = $request->get('invoice_id');
        
        // Build query to include invoices with specific statuses OR the specific invoice_id
        $query = Invoice::where(function($q) use ($customerId, $invoiceId) {
            $q->where('customer_id', $customerId)
              ->whereIn('status', ['final', 'partially_paid', 'paid']);
            
            // If a specific invoice_id is provided, include it even if it's paid or doesn't match customer
            if ($invoiceId) {
                $q->orWhere('id', $invoiceId);
            }
        });
        
        $invoices = $query->orderBy('created_at', 'desc')->get();

        return response()->json($invoices->map(function ($invoice) {
            return [
                'id' => $invoice->id,
                'invoice_number' => $invoice->invoice_number,
                'due_amount' => $invoice->due_amount,
                'formatted_due_amount' => format_currency($invoice->due_amount),
            ];
        }));
    }

    public function getPurchaseInvoices($supplierId, Request $request)
    {
        $purchaseInvoiceId = $request->get('purchase_invoice_id');
        
        // Build query to include purchase invoices with specific statuses OR the specific purchase_invoice_id
        $query = PurchaseInvoice::where(function($q) use ($supplierId, $purchaseInvoiceId) {
            $q->where('supplier_id', $supplierId)
              ->whereIn('status', ['confirmed', 'partially_paid', 'paid']);
            
            // If a specific purchase_invoice_id is provided, include it even if it's paid or doesn't match supplier
            if ($purchaseInvoiceId) {
                $q->orWhere('id', $purchaseInvoiceId);
            }
        });
        
        $purchaseInvoices = $query->orderBy('created_at', 'desc')->get();

        return response()->json($purchaseInvoices->map(function ($purchaseInvoice) {
            return [
                'id' => $purchaseInvoice->id,
                'invoice_number' => $purchaseInvoice->invoice_number,
                'amount_remaining' => $purchaseInvoice->amount_remaining,
                'formatted_amount_remaining' => format_currency($purchaseInvoice->amount_remaining),
            ];
        }));
    }
}
