<?php

namespace App\Http\Controllers;

use App\Models\PurchaseInvoice;
use App\Models\PurchaseInvoiceItem;
use App\Models\Product;
use App\Models\Setting;
use App\Models\Supplier;
use App\Services\StockManager;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use App\Traits\Sortable;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\PurchaseInvoicesExport;
use App\Imports\PurchaseInvoicesImport;

class PurchaseInvoiceController extends Controller
{
    use Sortable;

    public function index(Request $request)
    {
        $query = PurchaseInvoice::with(['supplier', 'user']);

        if ($request->filled('search')) {
            $query->where('invoice_number', 'like', '%' . $request->search . '%')
                ->orWhereHas('supplier', function ($q) use ($request) {
                    $q->where('name', 'like', '%' . $request->search . '%');
                });
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('supplier_id')) {
            $query->where('supplier_id', $request->supplier_id);
        }

        if ($request->filled('date_from')) {
            $query->where('purchase_date', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->where('purchase_date', '<=', $request->date_to);
        }

        $allowedColumns = ['invoice_number', 'purchase_date', 'total_amount', 'amount_paid', 'amount_remaining', 'status', 'created_at'];
        $sort = $request->get('sort', 'created_at');
        $direction = $request->get('direction', 'desc');

        if (in_array($sort, $allowedColumns)) {
            $query->orderBy($sort, $direction);
        } else {
            $query->latest();
        }

        $itemsPerPage = Setting::getItemsPerPage(20);
        $invoices = $query->paginate($itemsPerPage)->appends($request->query());
        $suppliers = Supplier::all();

        // Calculate supplier balance if filtered by supplier
        $supplierBalance = null;
        if ($request->filled('supplier_id')) {
            $supplier = Supplier::with('purchaseInvoices')->find($request->supplier_id);
            if ($supplier) {
                $supplierBalance = $supplier->calculateCurrentBalance();
            }
        }

        return view('purchase-invoices.index', compact('invoices', 'suppliers', 'sort', 'direction', 'supplierBalance'));
    }

    public function create()
    {
        $suppliers = Supplier::all();
        $products = Product::all();
        $warehouses = StockManager::isMultipleWarehouses() 
            ? \App\Models\Warehouse::active()->get() 
            : collect();

        return view('purchase-invoices.create', compact('suppliers', 'products', 'warehouses'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'supplier_id' => 'required|exists:suppliers,id',
            'invoice_number' => 'nullable|string|unique:purchase_invoices,invoice_number',
            'purchase_date' => 'required|date',
            'status' => 'required|in:draft,confirmed,paid,partially_paid',
            'discount' => 'nullable|numeric|min:0',
            'notes' => 'nullable|string',
            'supplier_invoice_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:10240',
            'warehouse_id' => 'nullable|exists:warehouses,id',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.purchase_price' => 'required|numeric|min:0',
            'items.*.tax' => 'nullable|numeric|min:0',
            'items.*.batch_number' => 'nullable|string|max:255',
            'items.*.expiry_date' => 'nullable|date',
        ]);

        DB::beginTransaction();
        try {
            $filePath = null;
            if ($request->hasFile('supplier_invoice_file')) {
                $filePath = $request->file('supplier_invoice_file')->store('supplier-invoices', 'public');
            }

            $invoice = PurchaseInvoice::create([
                'supplier_id' => $validated['supplier_id'],
                'invoice_number' => $validated['invoice_number'] ?? null,
                'purchase_date' => $validated['purchase_date'],
                'status' => $validated['status'],
                'discount' => $validated['discount'] ?? 0,
                'notes' => $validated['notes'] ?? null,
                'supplier_invoice_file' => $filePath,
                'user_id' => auth()->id(),
                'warehouse_id' => $validated['warehouse_id'] ?? null,
            ]);

            $subtotal = 0;
            $totalTax = 0;

            foreach ($validated['items'] as $itemData) {
                $itemTotal = ($itemData['quantity'] * $itemData['purchase_price']) + ($itemData['tax'] ?? 0);
                $subtotal += $itemTotal;

                PurchaseInvoiceItem::create([
                    'purchase_invoice_id' => $invoice->id,
                    'product_id' => $itemData['product_id'],
                    'quantity' => $itemData['quantity'],
                    'purchase_price' => $itemData['purchase_price'],
                    'tax' => $itemData['tax'] ?? 0,
                    'total' => $itemTotal,
                    'batch_number' => $itemData['batch_number'] ?? null,
                    'expiry_date' => $itemData['expiry_date'] ?? null,
                ]);

                $totalTax += $itemData['tax'] ?? 0;
            }

            $invoice->subtotal = $subtotal;
            $invoice->tax = $totalTax;
            $invoice->updateTotals();

            // If confirmed, update stock and create batches
            if ($invoice->status === 'confirmed') {
                // Get warehouse ID from invoice or use default
                $warehouseId = $invoice->warehouse_id;
                if (!$warehouseId && StockManager::isMultipleWarehouses()) {
                    $warehouseId = StockManager::getDefaultWarehouse()?->id;
                }
                
                foreach ($invoice->items as $item) {
                    $product = $item->product;
                    
                    // Update stock using StockManager
                    StockManager::updateStock(
                        $product,
                        $item->quantity,
                        'IN',
                        $warehouseId,
                        \App\Models\PurchaseInvoice::class,
                        $invoice->id,
                        "Purchase Invoice #{$invoice->invoice_number}"
                    );
                    $product->purchase_price = $item->purchase_price; // Update purchase price
                    $product->save();

                    // Create batch for tracking
                    if ($item->batch_number) {
                        \App\Models\ProductBatch::create([
                            'product_id' => $item->product_id,
                            'purchase_invoice_id' => $invoice->id,
                            'supplier_id' => $invoice->supplier_id,
                            'batch_number' => $item->batch_number,
                            'expiry_date' => $item->expiry_date,
                            'initial_quantity' => $item->quantity,
                            'current_quantity' => $item->quantity,
                            'purchase_price' => $item->purchase_price,
                            'purchase_date' => $invoice->purchase_date,
                        ]);
                    }
                }
            }

            DB::commit();

            return redirect()->route('purchase-invoices.show', $invoice)
                ->with('success', trans('messages.purchase_invoice_created'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withInput()->with('error', trans('messages.error_occurred') . ': ' . $e->getMessage());
        }
    }

    public function show(PurchaseInvoice $purchaseInvoice)
    {
        $purchaseInvoice->load(['supplier', 'items.product', 'payments.user', 'user']);

        return view('purchase-invoices.show', compact('purchaseInvoice'));
    }

    public function edit(PurchaseInvoice $purchaseInvoice)
    {
        if (!$purchaseInvoice->isDraft()) {
            return redirect()->route('purchase-invoices.show', $purchaseInvoice)
                ->with('error', trans('messages.cannot_edit_confirmed_invoice'));
        }

        $suppliers = Supplier::all();
        $products = Product::all();
        $warehouses = StockManager::isMultipleWarehouses() 
            ? \App\Models\Warehouse::active()->get() 
            : collect();
        $purchaseInvoice->load('items');

        return view('purchase-invoices.edit', compact('purchaseInvoice', 'suppliers', 'products', 'warehouses'));
    }

    public function update(Request $request, PurchaseInvoice $purchaseInvoice)
    {
        if (!$purchaseInvoice->isDraft()) {
            return redirect()->route('purchase-invoices.show', $purchaseInvoice)
                ->with('error', trans('messages.cannot_edit_confirmed_invoice'));
        }

        $validated = $request->validate([
            'supplier_id' => 'required|exists:suppliers,id',
            'invoice_number' => 'nullable|string|unique:purchase_invoices,invoice_number,' . $purchaseInvoice->id,
            'purchase_date' => 'required|date',
            'status' => 'required|in:draft,confirmed,paid,partially_paid',
            'discount' => 'nullable|numeric|min:0',
            'notes' => 'nullable|string',
            'supplier_invoice_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:10240',
            'warehouse_id' => 'nullable|exists:warehouses,id',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.purchase_price' => 'required|numeric|min:0',
            'items.*.tax' => 'nullable|numeric|min:0',
            'items.*.batch_number' => 'nullable|string|max:255',
            'items.*.expiry_date' => 'nullable|date',
        ]);

        DB::beginTransaction();
        try {
            if ($request->hasFile('supplier_invoice_file')) {
                if ($purchaseInvoice->supplier_invoice_file) {
                    Storage::disk('public')->delete($purchaseInvoice->supplier_invoice_file);
                }
                $purchaseInvoice->supplier_invoice_file = $request->file('supplier_invoice_file')->store('supplier-invoices', 'public');
            }

            $purchaseInvoice->update([
                'supplier_id' => $validated['supplier_id'],
                'invoice_number' => $validated['invoice_number'] ?? $purchaseInvoice->invoice_number,
                'purchase_date' => $validated['purchase_date'],
                'status' => $validated['status'],
                'discount' => $validated['discount'] ?? 0,
                'notes' => $validated['notes'] ?? null,
                'warehouse_id' => $validated['warehouse_id'] ?? null,
            ]);

            // Delete old items
            $purchaseInvoice->items()->delete();

            $subtotal = 0;
            $totalTax = 0;

            foreach ($validated['items'] as $itemData) {
                $itemTotal = ($itemData['quantity'] * $itemData['purchase_price']) + ($itemData['tax'] ?? 0);
                $subtotal += $itemTotal;

                PurchaseInvoiceItem::create([
                    'purchase_invoice_id' => $purchaseInvoice->id,
                    'product_id' => $itemData['product_id'],
                    'quantity' => $itemData['quantity'],
                    'purchase_price' => $itemData['purchase_price'],
                    'tax' => $itemData['tax'] ?? 0,
                    'total' => $itemTotal,
                    'batch_number' => $itemData['batch_number'] ?? null,
                    'expiry_date' => $itemData['expiry_date'] ?? null,
                ]);

                $totalTax += $itemData['tax'] ?? 0;
            }

            $purchaseInvoice->subtotal = $subtotal;
            $purchaseInvoice->tax = $totalTax;
            $purchaseInvoice->updateTotals();

            DB::commit();

            return redirect()->route('purchase-invoices.show', $purchaseInvoice)
                ->with('success', trans('messages.purchase_invoice_updated'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withInput()->with('error', trans('messages.error_occurred') . ': ' . $e->getMessage());
        }
    }

    public function destroy(PurchaseInvoice $purchaseInvoice)
    {
        // Check permissions - Only Admin and Super Admin can delete
        $user = auth()->user();
        if (!$user->hasAnyRole(['Admin', 'Super Admin'])) {
            abort(403, trans('messages.unauthorized_action'));
        }

        DB::beginTransaction();
        try {
            // Load relationships
            $purchaseInvoice->load('items.product');

            // If invoice was confirmed (not draft or returned), reverse stock changes
            // Only reverse stock if invoice was confirmed/paid/partially_paid (stock was increased)
            // Do NOT reverse for returned invoices (stock was already decreased)
            if (!$purchaseInvoice->isReturned() && ($purchaseInvoice->isConfirmed() || $purchaseInvoice->status === 'paid' || $purchaseInvoice->status === 'partially_paid')) {
                // Get warehouse ID from invoice or use default
                $warehouseId = $purchaseInvoice->warehouse_id;
                if (!$warehouseId && StockManager::isMultipleWarehouses()) {
                    $warehouseId = StockManager::getDefaultWarehouse()?->id;
                }
                
                foreach ($purchaseInvoice->items as $item) {
                    $product = $item->product;
                    if ($product) {
                        // Reverse stock using StockManager
                        StockManager::updateStock(
                            $product,
                            $item->quantity,
                            'OUT',
                            $warehouseId,
                            PurchaseInvoice::class,
                            $purchaseInvoice->id,
                            "Stock reversed - Purchase Invoice #{$purchaseInvoice->invoice_number} deleted"
                        );
                    }
                }
            }

            // Delete supplier invoice file if exists
            if ($purchaseInvoice->supplier_invoice_file) {
                Storage::disk('public')->delete($purchaseInvoice->supplier_invoice_file);
            }

            // Delete related payments
            $purchaseInvoice->payments()->delete();

            // Delete related items
            $purchaseInvoice->items()->delete();

            // Log deletion in audit_logs
            $invoiceData = $purchaseInvoice->toArray();
            \App\Models\AuditLog::create([
                'action' => 'delete',
                'model_type' => PurchaseInvoice::class,
                'model_id' => $purchaseInvoice->id,
                'old_values' => $invoiceData,
                'new_values' => null,
                'user_id' => auth()->id(),
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
            ]);

            // Delete the purchase invoice
            $purchaseInvoice->delete();

            DB::commit();

            return redirect()->route('purchase-invoices.index')
                ->with('success', trans('messages.purchase_invoice_deleted'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.error_occurred') . ': ' . $e->getMessage());
        }
    }

    public function confirm(PurchaseInvoice $purchaseInvoice)
    {
        if (!$purchaseInvoice->isDraft()) {
            return back()->with('error', trans('messages.invoice_already_confirmed'));
        }

        DB::beginTransaction();
        try {
            $purchaseInvoice->status = 'confirmed';
            $purchaseInvoice->save();

            // Get warehouse ID from invoice or use default
            $warehouseId = $purchaseInvoice->warehouse_id;
            if (!$warehouseId && StockManager::isMultipleWarehouses()) {
                $warehouseId = StockManager::getDefaultWarehouse()?->id;
            }
            
            // Update stock and create batches
            foreach ($purchaseInvoice->items as $item) {
                $product = $item->product;
                
                // Update stock using StockManager
                StockManager::updateStock(
                    $product,
                    $item->quantity,
                    'IN',
                    $warehouseId,
                    PurchaseInvoice::class,
                    $purchaseInvoice->id,
                    "Purchase Invoice #{$purchaseInvoice->invoice_number} confirmed"
                );
                
                $product->purchase_price = $item->purchase_price;
                $product->save();

                // Create batch for tracking
                if ($item->batch_number) {
                    \App\Models\ProductBatch::create([
                        'product_id' => $item->product_id,
                        'purchase_invoice_id' => $purchaseInvoice->id,
                        'supplier_id' => $purchaseInvoice->supplier_id,
                        'batch_number' => $item->batch_number,
                        'expiry_date' => $item->expiry_date,
                        'initial_quantity' => $item->quantity,
                        'current_quantity' => $item->quantity,
                        'purchase_price' => $item->purchase_price,
                        'purchase_date' => $purchaseInvoice->purchase_date,
                    ]);
                }
            }

            DB::commit();

            return redirect()->route('purchase-invoices.show', $purchaseInvoice)
                ->with('success', trans('messages.purchase_invoice_confirmed'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.error_occurred') . ': ' . $e->getMessage());
        }
    }

    public function return(PurchaseInvoice $purchaseInvoice)
    {
        if ($purchaseInvoice->isReturned()) {
            return back()->with('error', trans('messages.invoice_already_returned'));
        }

        if (!$purchaseInvoice->isConfirmed()) {
            return back()->with('error', trans('messages.can_only_return_confirmed_invoices'));
        }

        DB::beginTransaction();
        try {
            $purchaseInvoice->status = 'returned';
            $purchaseInvoice->save();

            // Get warehouse ID from invoice or use default
            $warehouseId = $purchaseInvoice->warehouse_id;
            if (!$warehouseId && StockManager::isMultipleWarehouses()) {
                $warehouseId = StockManager::getDefaultWarehouse()?->id;
            }
            
            // Decrease stock
            foreach ($purchaseInvoice->items as $item) {
                $product = $item->product;
                
                // Update stock using StockManager
                StockManager::updateStock(
                    $product,
                    $item->quantity,
                    'OUT',
                    $warehouseId,
                    PurchaseInvoice::class,
                    $purchaseInvoice->id,
                    "Purchase Invoice #{$purchaseInvoice->invoice_number} returned"
                );
            }

            DB::commit();

            return redirect()->route('purchase-invoices.show', $purchaseInvoice)
                ->with('success', trans('messages.purchase_invoice_returned'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.error_occurred') . ': ' . $e->getMessage());
        }
    }

    public function bulkDelete(Request $request)
    {
        // Check permissions - Only Admin and Super Admin can bulk delete
        $user = auth()->user();
        if (!$user->hasAnyRole(['Admin', 'Super Admin'])) {
            abort(403, trans('messages.unauthorized_action'));
        }

        $request->validate([
            'selected_ids' => 'required|array|min:1',
            'selected_ids.*' => 'exists:purchase_invoices,id',
        ]);

        DB::beginTransaction();
        try {
            $selectedIds = $request->selected_ids;
            $purchaseInvoices = PurchaseInvoice::with('items.product')->whereIn('id', $selectedIds)->get();
            
            // Check for confirmed invoices - cannot delete
            $confirmedInvoices = $purchaseInvoices->filter(function ($invoice) {
                return !$invoice->isDraft();
            });

            if ($confirmedInvoices->count() > 0) {
                DB::rollBack();
                return back()->with('error', trans('messages.cannot_delete_confirmed_invoice'));
            }
            
            foreach ($purchaseInvoices as $purchaseInvoice) {
                // Delete supplier invoice file if exists
                if ($purchaseInvoice->supplier_invoice_file) {
                    Storage::disk('public')->delete($purchaseInvoice->supplier_invoice_file);
                }

                // Delete related payments
                $purchaseInvoice->payments()->delete();

                // Delete related items
                $purchaseInvoice->items()->delete();

                // Log deletion in audit_logs
                $invoiceData = $purchaseInvoice->toArray();
                \App\Models\AuditLog::create([
                    'action' => 'bulk_delete',
                    'model_type' => PurchaseInvoice::class,
                    'model_id' => $purchaseInvoice->id,
                    'old_values' => $invoiceData,
                    'new_values' => null,
                    'user_id' => auth()->id(),
                    'ip_address' => request()->ip(),
                    'user_agent' => request()->userAgent(),
                ]);
            }

            // Delete all purchase invoices
            PurchaseInvoice::whereIn('id', $selectedIds)->delete();

            DB::commit();

            return redirect()->route('purchase-invoices.index')
                ->with('success', trans('messages.bulk_delete_success', ['count' => count($selectedIds)]));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.bulk_delete_error') . ': ' . $e->getMessage());
        }
    }

    public function pdf(PurchaseInvoice $purchaseInvoice)
    {
        $purchaseInvoice->load(['supplier', 'items.product', 'user']);
        // استخدام صفحة الطباعة لجميع اللغات
        return view('purchase-invoices.print', compact('purchaseInvoice'));
    }

    /**
     * طباعة فاتورة الشراء
     */
    public function print(PurchaseInvoice $purchaseInvoice)
    {
        $purchaseInvoice->load(['supplier', 'items.product', 'user']);
        return view('purchase-invoices.print', compact('purchaseInvoice'));
    }

    /**
     * تصدير Bon de Commande (أمر الشراء)
     */
    public function bonDeCommande(PurchaseInvoice $purchaseInvoice)
    {
        $purchaseInvoice->load(['supplier', 'items.product', 'user']);
        // استخدام صفحة الطباعة لجميع اللغات
        return view('purchase-invoices.bon-commande-print', compact('purchaseInvoice'));
    }

    public function export(Request $request)
    {
        $query = PurchaseInvoice::with(['supplier', 'user']);

        // Apply same filters as index
        if ($request->filled('search')) {
            $query->where('invoice_number', 'like', '%' . $request->search . '%')
                ->orWhereHas('supplier', function ($q) use ($request) {
                    $q->where('name', 'like', '%' . $request->search . '%');
                });
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('supplier_id')) {
            $query->where('supplier_id', $request->supplier_id);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('purchase_date', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('purchase_date', '<=', $request->date_to);
        }

        $purchaseInvoices = $query->get();
        $filename = 'purchase_invoices_' . date('Y-m-d_His') . '.xlsx';

        return Excel::download(new PurchaseInvoicesExport($purchaseInvoices), $filename);
    }

    public function importForm()
    {
        return view('purchase-invoices.import');
    }

    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:xlsx,xls|max:10240',
        ]);

        try {
            $import = new PurchaseInvoicesImport();
            Excel::import($import, $request->file('file'));

            $imported = $import->getImported();
            $failed = $import->getFailed();
            $errors = $import->getErrors();

            if ($imported === 0 && $failed === 0) {
                return back()->with('error', trans('messages.excel_file_empty'));
            }

            $message = trans('messages.import_completed', [
                'imported' => $imported,
                'failed' => $failed,
            ]);

            if ($failed > 0 && !empty($errors)) {
                return back()->with('warning', $message)->with('import_errors', $errors);
            }

            return back()->with('success', $message);

        } catch (\Maatwebsite\Excel\Validators\ValidationException $e) {
            $failures = $e->failures();
            $errors = [];
            
            foreach ($failures as $failure) {
                $errors[] = [
                    'row' => $failure->row(),
                    'name' => $failure->values()['invoice_number'] ?? 'N/A',
                    'error' => implode(', ', $failure->errors()),
                ];
            }
            
            return back()->with('error', trans('messages.import_error'))
                        ->with('import_errors', $errors);
                        
        } catch (\Maatwebsite\Excel\Exceptions\SheetNotFoundException $e) {
            return back()->with('error', trans('messages.import_error') . ': ' . 'Sheet not found in Excel file.');
        } catch (\PhpOffice\PhpSpreadsheet\Reader\Exception $e) {
            return back()->with('error', trans('messages.import_error') . ': ' . 'Error reading Excel file. Please check if the file is valid.');
        } catch (\Exception $e) {
            \Log::error('Purchase Invoice import error: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            return back()->with('error', trans('messages.import_error') . ': ' . $e->getMessage());
        }
    }

    public function downloadTemplate()
    {
        $data = [
            ['invoice_number', 'supplier_id', 'purchase_date', 'status', 'subtotal', 'tax', 'discount', 'total_amount', 'amount_paid', 'amount_remaining', 'notes'],
            ['FDA-000001', '1', '2024-01-01', 'draft', '500.00', '50.00', '0.00', '550.00', '0.00', '550.00', 'Notes here'],
            ['FDA-000002', '2', '2024-01-02', 'confirmed', '1000.00', '100.00', '50.00', '1050.00', '500.00', '550.00', ''],
        ];

        $filename = 'purchase_invoice_import_template_' . date('Y-m-d') . '.xlsx';
        
        $export = new class($data) implements \Maatwebsite\Excel\Concerns\FromArray {
            protected $data;
            
            public function __construct($data) {
                $this->data = $data;
            }
            
            public function array(): array {
                return $this->data;
            }
        };
        
        return Excel::download($export, $filename);
    }
}

