<?php

namespace App\Http\Controllers;

use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderItem;
use App\Models\Product;
use App\Models\Supplier;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class PurchaseOrderController extends Controller
{
    public function index(Request $request)
    {
        $query = PurchaseOrder::with(['supplier', 'creator']);

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('order_number', 'like', "%{$search}%")
                  ->orWhereHas('supplier', function ($q) use ($search) {
                      $q->where('name', 'like', "%{$search}%");
                  });
            });
        }

        $itemsPerPage = Setting::getItemsPerPage(20);
        $orders = $query->latest()->paginate($itemsPerPage)->appends($request->query());
        $suppliers = Supplier::all();

        return view('purchase-orders.index', compact('orders', 'suppliers'));
    }

    public function create()
    {
        $suppliers = Supplier::all();
        $products = Product::all();
        return view('purchase-orders.create', compact('suppliers', 'products'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'supplier_id' => 'nullable|exists:suppliers,id',
            'expected_delivery_date' => 'nullable|date',
            'notes' => 'nullable|string',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.unit_price' => 'required|numeric|min:0',
        ]);

        DB::beginTransaction();
        try {
            $order = PurchaseOrder::create([
                'supplier_id' => $validated['supplier_id'] ?? null,
                'status' => 'draft',
                'expected_delivery_date' => $validated['expected_delivery_date'] ?? null,
                'notes' => $validated['notes'] ?? null,
                'generation_type' => 'manual',
                'created_by' => auth()->id(),
            ]);

            foreach ($validated['items'] as $item) {
                PurchaseOrderItem::create([
                    'purchase_order_id' => $order->id,
                    'product_id' => $item['product_id'],
                    'quantity' => $item['quantity'],
                    'unit_price' => $item['unit_price'],
                ]);
            }

            $order->updateTotals();

            DB::commit();
            return redirect()->route('purchase-orders.show', $order)
                ->with('success', trans('messages.purchase_order_created'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.error_occurred') . ': ' . $e->getMessage());
        }
    }

    public function show(PurchaseOrder $purchaseOrder)
    {
        $purchaseOrder->load(['items.product', 'supplier', 'creator', 'approver']);
        return view('purchase-orders.show', compact('purchaseOrder'));
    }

    public function edit(PurchaseOrder $purchaseOrder)
    {
        if (!$purchaseOrder->isDraft()) {
            return back()->with('error', trans('messages.cannot_edit_non_draft_order'));
        }

        $purchaseOrder->load('items.product');
        $suppliers = Supplier::all();
        $products = Product::all();
        return view('purchase-orders.edit', compact('purchaseOrder', 'suppliers', 'products'));
    }

    public function update(Request $request, PurchaseOrder $purchaseOrder)
    {
        if (!$purchaseOrder->isDraft()) {
            return back()->with('error', trans('messages.cannot_edit_non_draft_order'));
        }

        $validated = $request->validate([
            'supplier_id' => 'nullable|exists:suppliers,id',
            'expected_delivery_date' => 'nullable|date',
            'notes' => 'nullable|string',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.unit_price' => 'required|numeric|min:0',
        ]);

        DB::beginTransaction();
        try {
            $purchaseOrder->update([
                'supplier_id' => $validated['supplier_id'] ?? null,
                'expected_delivery_date' => $validated['expected_delivery_date'] ?? null,
                'notes' => $validated['notes'] ?? null,
            ]);

            $purchaseOrder->items()->delete();

            foreach ($validated['items'] as $item) {
                PurchaseOrderItem::create([
                    'purchase_order_id' => $purchaseOrder->id,
                    'product_id' => $item['product_id'],
                    'quantity' => $item['quantity'],
                    'unit_price' => $item['unit_price'],
                ]);
            }

            $purchaseOrder->updateTotals();

            DB::commit();
            return redirect()->route('purchase-orders.show', $purchaseOrder)
                ->with('success', trans('messages.purchase_order_updated'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.error_occurred') . ': ' . $e->getMessage());
        }
    }

    public function approve(PurchaseOrder $purchaseOrder)
    {
        $purchaseOrder->approve(auth()->id());
        return back()->with('success', trans('messages.purchase_order_approved'));
    }

    public function convertToPurchaseInvoice(PurchaseOrder $purchaseOrder)
    {
        if (!$purchaseOrder->supplier_id) {
            return back()->with('error', trans('messages.purchase_order_must_have_supplier'));
        }

        DB::beginTransaction();
        try {
            $purchaseOrder->load(['items.product', 'supplier']);

            // Create purchase invoice (invoice_number will be auto-generated by model boot)
            $purchaseInvoice = \App\Models\PurchaseInvoice::create([
                'supplier_id' => $purchaseOrder->supplier_id,
                'invoice_number' => '', // Will be auto-generated by model boot
                'purchase_date' => $purchaseOrder->expected_delivery_date ?? now(),
                'status' => 'draft',
                'discount' => 0,
                'notes' => trans('messages.converted_from_purchase_order') . ': ' . $purchaseOrder->order_number,
                'user_id' => auth()->id(),
            ]);

            $subtotal = 0;
            $totalTax = 0;

            // Create invoice items from order items
            foreach ($purchaseOrder->items as $orderItem) {
                $itemTotal = $orderItem->quantity * $orderItem->unit_price;
                $subtotal += $itemTotal;

                \App\Models\PurchaseInvoiceItem::create([
                    'purchase_invoice_id' => $purchaseInvoice->id,
                    'product_id' => $orderItem->product_id,
                    'quantity' => $orderItem->quantity,
                    'purchase_price' => $orderItem->unit_price,
                    'tax' => 0,
                    'total' => $itemTotal,
                ]);

                $totalTax += 0;
            }

            $purchaseInvoice->subtotal = $subtotal;
            $purchaseInvoice->tax = $totalTax;
            $purchaseInvoice->updateTotals();

            // Update purchase order status
            $purchaseOrder->update(['status' => 'completed']);

            DB::commit();
            return redirect()->route('purchase-invoices.show', $purchaseInvoice)
                ->with('success', trans('messages.purchase_order_converted_to_invoice'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.error_occurred') . ': ' . $e->getMessage());
        }
    }

    public function bulkDelete(Request $request)
    {
        $request->validate([
            'selected_ids' => 'required|array|min:1',
            'selected_ids.*' => 'exists:purchase_orders,id',
        ]);

        DB::beginTransaction();
        try {
            $selectedIds = $request->selected_ids;
            $orders = PurchaseOrder::whereIn('id', $selectedIds)->where('status', 'draft')->get();
            
            $deletedCount = 0;
            foreach ($orders as $order) {
                $order->items()->delete();
                $order->delete();
                $deletedCount++;
            }

            DB::commit();
            return redirect()->route('purchase-orders.index')
                ->with('success', trans('messages.bulk_delete_success', ['count' => $deletedCount]));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.bulk_delete_error') . ': ' . $e->getMessage());
        }
    }

    public function destroy(PurchaseOrder $purchaseOrder)
    {
        if (!$purchaseOrder->isDraft()) {
            return back()->with('error', trans('messages.cannot_delete_non_draft_order'));
        }

        $purchaseOrder->delete();
        return redirect()->route('purchase-orders.index')
            ->with('success', trans('messages.purchase_order_deleted'));
    }
}
