<?php

namespace App\Http\Controllers;

use App\Models\QualityComplaint;
use App\Models\Product;
use App\Models\Customer;
use App\Models\Supplier;
use App\Models\Invoice;
use App\Models\PurchaseInvoice;
use App\Models\Setting;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use App\Traits\Sortable;
use App\Traits\ChecksPermissions;

class QualityComplaintController extends Controller
{
    use Sortable, ChecksPermissions;

    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index(Request $request)
    {
        $this->checkPermission('quality.complaints.view');
        
        $query = QualityComplaint::with(['product', 'customer', 'supplier', 'invoice', 'purchaseInvoice', 'reportedBy', 'assignedTo']);

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('complaint_number', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%")
                  ->orWhere('complainant_name', 'like', "%{$search}%")
                  ->orWhere('complainant_email', 'like', "%{$search}%")
                  ->orWhereHas('product', function ($q) use ($search) {
                      $q->where('name', 'like', "%{$search}%")
                        ->orWhere('name_ar', 'like', "%{$search}%");
                  });
            });
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('severity')) {
            $query->where('severity', $request->severity);
        }

        if ($request->filled('complaint_type')) {
            $query->where('complaint_type', $request->complaint_type);
        }

        if ($request->filled('product_id')) {
            $query->where('product_id', $request->product_id);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('reported_date', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('reported_date', '<=', $request->date_to);
        }

        $allowedColumns = ['complaint_number', 'reported_date', 'status', 'severity', 'created_at'];
        $sort = $request->get('sort', 'created_at');
        $direction = $request->get('direction', 'desc');

        if (in_array($sort, $allowedColumns)) {
            $query->orderBy($sort, $direction);
        } else {
            $query->latest();
        }

        $itemsPerPage = Setting::getItemsPerPage(20);
        $complaints = $query->paginate($itemsPerPage)->appends($request->query());
        $products = Product::all();
        $customers = Customer::all();
        $suppliers = Supplier::all();
        $users = User::all();

        return view('quality.complaints.index', compact('complaints', 'products', 'customers', 'suppliers', 'users', 'sort', 'direction'));
    }

    public function create(Request $request)
    {
        $this->checkPermission('quality.complaints.create');
        
        $products = Product::all();
        $customers = Customer::all();
        $suppliers = Supplier::all();
        $invoices = Invoice::whereIn('status', ['paid', 'partially_paid', 'final'])->get();
        $purchaseInvoices = PurchaseInvoice::whereIn('status', ['confirmed', 'paid', 'partially_paid'])->get();
        $users = User::all();
        
        $productId = $request->get('product_id');
        $customerId = $request->get('customer_id');
        $invoiceId = $request->get('invoice_id');

        return view('quality.complaints.create', compact('products', 'customers', 'suppliers', 'invoices', 'purchaseInvoices', 'users', 'productId', 'customerId', 'invoiceId'));
    }

    public function store(Request $request)
    {
        $this->checkPermission('quality.complaints.create');

        $validated = $request->validate([
            'complaint_type' => 'required|in:customer,supplier,internal,other',
            'product_id' => 'nullable|exists:products,id',
            'customer_id' => 'nullable|exists:customers,id',
            'supplier_id' => 'nullable|exists:suppliers,id',
            'invoice_id' => 'nullable|exists:invoices,id',
            'purchase_invoice_id' => 'nullable|exists:purchase_invoices,id',
            'complainant_name' => 'nullable|string|max:255',
            'complainant_email' => 'nullable|email|max:255',
            'complainant_phone' => 'nullable|string|max:50',
            'description' => 'required|string',
            'description_ar' => 'nullable|string',
            'severity' => 'required|in:low,medium,high,critical',
            'status' => 'required|in:new,in_progress,resolved,closed,rejected',
            'assigned_to' => 'nullable|exists:users,id',
            'reported_date' => 'required|date',
            'images' => 'nullable|array',
            'images.*' => 'image|max:2048',
        ]);

        DB::beginTransaction();
        try {
            $imagePaths = [];
            if ($request->hasFile('images')) {
                foreach ($request->file('images') as $image) {
                    $imagePaths[] = $image->store('quality/complaints', 'public');
                }
            }
            $validated['images'] = $imagePaths;
            $validated['reported_by'] = auth()->id();

            QualityComplaint::create($validated);

            DB::commit();

            return redirect()->route('quality.complaints.index')
                ->with('success', trans('messages.complaint_created'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withInput()->with('error', trans('messages.error_occurred') . ': ' . $e->getMessage());
        }
    }

    public function show(QualityComplaint $complaint)
    {
        $this->checkPermission('quality.complaints.view');
        
        $complaint->load(['product', 'customer', 'supplier', 'invoice', 'purchaseInvoice', 'reportedBy', 'assignedTo']);

        return view('quality.complaints.show', compact('complaint'));
    }

    public function edit(QualityComplaint $complaint)
    {
        $this->checkPermission('quality.complaints.update');
        
        $products = Product::all();
        $customers = Customer::all();
        $suppliers = Supplier::all();
        $invoices = Invoice::whereIn('status', ['paid', 'partially_paid', 'final'])->get();
        $purchaseInvoices = PurchaseInvoice::whereIn('status', ['confirmed', 'paid', 'partially_paid'])->get();
        $users = User::all();

        return view('quality.complaints.edit', compact('complaint', 'products', 'customers', 'suppliers', 'invoices', 'purchaseInvoices', 'users'));
    }

    public function update(Request $request, QualityComplaint $complaint)
    {
        $this->checkPermission('quality.complaints.update');

        $validated = $request->validate([
            'complaint_type' => 'required|in:customer,supplier,internal,other',
            'product_id' => 'nullable|exists:products,id',
            'customer_id' => 'nullable|exists:customers,id',
            'supplier_id' => 'nullable|exists:suppliers,id',
            'invoice_id' => 'nullable|exists:invoices,id',
            'purchase_invoice_id' => 'nullable|exists:purchase_invoices,id',
            'complainant_name' => 'nullable|string|max:255',
            'complainant_email' => 'nullable|email|max:255',
            'complainant_phone' => 'nullable|string|max:50',
            'description' => 'required|string',
            'description_ar' => 'nullable|string',
            'severity' => 'required|in:low,medium,high,critical',
            'status' => 'required|in:new,in_progress,resolved,closed,rejected',
            'assigned_to' => 'nullable|exists:users,id',
            'reported_date' => 'required|date',
            'resolved_date' => 'nullable|date',
            'resolution_notes' => 'nullable|string',
            'images' => 'nullable|array',
            'images.*' => 'image|max:2048',
        ]);

        DB::beginTransaction();
        try {
            // Handle image uploads
            $existingImages = $complaint->images ?? [];
            $newImagePaths = [];
            
            if ($request->hasFile('images')) {
                foreach ($request->file('images') as $image) {
                    $newImagePaths[] = $image->store('quality/complaints', 'public');
                }
            }

            // Keep existing images if not deleted
            if ($request->filled('existing_images')) {
                $existingImages = array_intersect($existingImages, $request->existing_images);
            } else {
                // Delete all existing images if none selected
                foreach ($existingImages as $image) {
                    Storage::disk('public')->delete($image);
                }
                $existingImages = [];
            }

            $validated['images'] = array_merge($existingImages, $newImagePaths);

            // Set resolved_date if status is resolved or closed
            if (in_array($validated['status'], ['resolved', 'closed']) && !$complaint->resolved_date) {
                $validated['resolved_date'] = now();
            }

            $complaint->update($validated);

            DB::commit();

            return redirect()->route('quality.complaints.show', $complaint)
                ->with('success', trans('messages.complaint_updated'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withInput()->with('error', trans('messages.error_occurred') . ': ' . $e->getMessage());
        }
    }

    public function destroy(QualityComplaint $complaint)
    {
        $this->checkPermission('quality.complaints.delete');

        DB::beginTransaction();
        try {
            // Delete images
            if ($complaint->images) {
                foreach ($complaint->images as $image) {
                    Storage::disk('public')->delete($image);
                }
            }

            $complaint->delete();

            DB::commit();

            return redirect()->route('quality.complaints.index')
                ->with('success', trans('messages.complaint_deleted'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.error_occurred') . ': ' . $e->getMessage());
        }
    }
}
