<?php

namespace App\Http\Controllers;

use App\Models\Invoice;
use App\Models\InvoiceItem;
use App\Models\Product;
use App\Models\Setting;
use App\Models\Payment;
use App\Models\PurchaseInvoice;
use App\Models\SupplierPayment;
use App\Models\Customer;
use App\Models\Supplier;
use App\Models\StockMovement;
use App\Models\Treasury;
use App\Models\Warehouse;
use App\Services\StockManager;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;

class ReportController extends Controller
{
    public function sales(Request $request)
    {
        $query = Invoice::query();

        // Include all statuses except draft
        // Returned invoices will be shown in table but excluded from calculations
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        } else {
            $query->whereIn('status', ['paid', 'partially_paid', 'final', 'returned']);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        } else {
            $query->whereDate('created_at', '>=', now()->startOfMonth());
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        } else {
            $query->whereDate('created_at', '<=', now()->endOfMonth());
        }

        // Get all invoices for totals calculation (before pagination)
        // Create a separate query for calculations that excludes returned invoices
        $calculationQuery = Invoice::query();
        
        // Apply same filters as main query but always exclude returned
        if ($request->filled('status') && $request->status !== 'returned') {
            $calculationQuery->where('status', $request->status);
        } else {
            // If no status filter or filtering by returned, exclude returned from calculations
            $calculationQuery->whereIn('status', ['paid', 'partially_paid', 'final']);
        }
        
        // Always exclude returned invoices from calculations
        $calculationQuery->where('status', '!=', 'returned');
        
        if ($request->filled('date_from')) {
            $calculationQuery->whereDate('created_at', '>=', $request->date_from);
        } else {
            $calculationQuery->whereDate('created_at', '>=', now()->startOfMonth());
        }

        if ($request->filled('date_to')) {
            $calculationQuery->whereDate('created_at', '<=', $request->date_to);
        } else {
            $calculationQuery->whereDate('created_at', '<=', now()->endOfMonth());
        }
        
        $allInvoices = $calculationQuery->with(['customer', 'user', 'items.product'])
            ->latest()
            ->get();
        
        // Ensure status is correctly set based on due_amount
        foreach ($allInvoices as $invoice) {
            if ($invoice->status !== 'draft' && $invoice->status !== 'returned') {
                if ($invoice->due_amount <= 0) {
                    $invoice->status = 'paid';
                } elseif ($invoice->paid_amount > 0 && $invoice->due_amount > 0) {
                    $invoice->status = 'partially_paid';
                } elseif ($invoice->paid_amount == 0 && $invoice->status == 'final') {
                    // Keep as final if not paid yet
                }
            }
            
            // Calculate profit for each invoice
            $invoice->profit = $invoice->items->sum(function ($item) {
                $revenue = $item->total;
                $cost = $item->quantity * ($item->product->purchase_price ?? 0);
                return $revenue - $cost;
            });
        }
        
        // Calculate totals from all invoices (excluding returned)
        // $allInvoices already excludes returned invoices from the query
        $totalSales = $allInvoices->sum('total');
        $totalPaid = $allInvoices->sum('paid_amount');
        $totalDue = $allInvoices->sum('due_amount');
        $totalProfit = $allInvoices->sum('profit');

        // Calculate daily sales excluding returned invoices
        $dailySales = $allInvoices->groupBy(function ($invoice) {
                return $invoice->created_at->format('Y-m-d');
            })->map(function ($group) {
                return [
                    'date' => $group->first()->created_at->format('Y-m-d'),
                    'count' => $group->count(),
                    'total' => $group->sum('total'),
                ];
            })->values();

        // Paginate the invoices for display (including returned for display only)
        $itemsPerPage = Setting::getItemsPerPage(20);
        $invoices = $query->with(['customer', 'user'])
            ->latest()
            ->paginate($itemsPerPage)
            ->appends($request->query());
        
        // Apply status corrections to paginated results
        foreach ($invoices as $invoice) {
            if ($invoice->status !== 'draft' && $invoice->status !== 'returned') {
                if ($invoice->due_amount <= 0) {
                    $invoice->status = 'paid';
                } elseif ($invoice->paid_amount > 0 && $invoice->due_amount > 0) {
                    $invoice->status = 'partially_paid';
                } elseif ($invoice->paid_amount == 0 && $invoice->status == 'final') {
                    // Keep as final if not paid yet
                }
            }
        }

        // Calculate profit for paginated invoices
        foreach ($invoices as $invoice) {
            $invoice->load('items.product');
            $invoice->profit = $invoice->items->sum(function ($item) {
                $revenue = $item->total;
                $cost = $item->quantity * ($item->product->purchase_price ?? 0);
                return $revenue - $cost;
            });
        }
        
        return view('reports.sales', compact('invoices', 'totalSales', 'totalPaid', 'totalDue', 'totalProfit', 'dailySales'));
    }

    public function products(Request $request)
    {
        $query = InvoiceItem::with(['product', 'invoice'])
            ->whereHas('invoice', function ($q) {
                $q->whereIn('status', ['paid', 'partially_paid', 'final']);
            });

        if ($request->filled('date_from')) {
            $query->whereHas('invoice', function ($q) use ($request) {
                $q->whereDate('created_at', '>=', $request->date_from);
            });
        }

        if ($request->filled('date_to')) {
            $query->whereHas('invoice', function ($q) use ($request) {
                $q->whereDate('created_at', '<=', $request->date_to);
            });
        }

        if ($request->filled('product_id')) {
            $query->where('product_id', $request->product_id);
        }

        $items = $query->get();

        $productSales = $items->groupBy('product_id')->map(function ($group) {
            $product = $group->first()->product;
            return [
                'product' => $product,
                'quantity_sold' => $group->sum('quantity'),
                'total_revenue' => $group->sum('total'),
                'total_cost' => $group->sum(function ($item) {
                    return $item->quantity * $item->product->purchase_price;
                }),
            ];
        })->values();

        // Paginate product sales for display
        $itemsPerPage = Setting::getItemsPerPage(20);
        $currentPage = $request->get('page', 1);
        $offset = ($currentPage - 1) * $itemsPerPage;
        $paginatedSales = $productSales->slice($offset, $itemsPerPage)->values();
        $productSalesPaginator = new \Illuminate\Pagination\LengthAwarePaginator(
            $paginatedSales,
            $productSales->count(),
            $itemsPerPage,
            $currentPage,
            ['path' => $request->url(), 'query' => $request->query()]
        );

        return view('reports.products', compact('productSalesPaginator'));
    }

    public function profit(Request $request)
    {
        $query = InvoiceItem::with(['product', 'invoice'])
            ->whereHas('invoice', function ($q) {
                $q->whereIn('status', ['paid', 'partially_paid', 'final']);
            });

        if ($request->filled('date_from')) {
            $query->whereHas('invoice', function ($q) use ($request) {
                $q->whereDate('created_at', '>=', $request->date_from);
            });
        }

        if ($request->filled('date_to')) {
            $query->whereHas('invoice', function ($q) use ($request) {
                $q->whereDate('created_at', '<=', $request->date_to);
            });
        }

        $items = $query->get();

        $totalRevenue = $items->sum('total');
        $totalCost = $items->sum(function ($item) {
            return $item->quantity * $item->product->purchase_price;
        });
        $totalProfit = $totalRevenue - $totalCost;
        $profitMargin = $totalRevenue > 0 ? ($totalProfit / $totalRevenue) * 100 : 0;

        $productProfits = $items->groupBy('product_id')->map(function ($group) {
            $product = $group->first()->product;
            $revenue = $group->sum('total');
            $cost = $group->sum(function ($item) {
                return $item->quantity * $item->product->purchase_price;
            });
            
            return [
                'product' => $product,
                'quantity_sold' => $group->sum('quantity'),
                'revenue' => $revenue,
                'cost' => $cost,
                'profit' => $revenue - $cost,
                'margin' => $revenue > 0 ? (($revenue - $cost) / $revenue) * 100 : 0,
            ];
        })->sortByDesc('profit')->values();

        // Paginate product profits for display
        $itemsPerPage = Setting::getItemsPerPage(20);
        $currentPage = $request->get('page', 1);
        $offset = ($currentPage - 1) * $itemsPerPage;
        $paginatedProfits = $productProfits->slice($offset, $itemsPerPage)->values();
        $productProfitsPaginator = new \Illuminate\Pagination\LengthAwarePaginator(
            $paginatedProfits,
            $productProfits->count(),
            $itemsPerPage,
            $currentPage,
            ['path' => $request->url(), 'query' => $request->query()]
        );

        return view('reports.profit', compact('totalRevenue', 'totalCost', 'totalProfit', 'profitMargin', 'productProfitsPaginator'));
    }

    public function stock(Request $request)
    {
        // Get warehouse ID if multiple warehouses mode
        $warehouseId = null;
        if (StockManager::isMultipleWarehouses()) {
            $warehouseId = $request->get('warehouse_id');
        }
        
        // Calculate totals from all products (before pagination)
        $allProducts = Product::with('category')->get();
        $totalValue = $allProducts->sum(function ($product) use ($warehouseId) {
            return StockManager::getStockValue($product, $warehouseId);
        });
        $lowStockCount = $allProducts->filter(function ($product) use ($warehouseId) {
            return $product->isLowStock($warehouseId);
        })->count();

        // Paginate products for display
        $query = Product::with('category');
        $itemsPerPage = Setting::getItemsPerPage(20);
        $products = $query->paginate($itemsPerPage)->appends($request->query());

        return view('reports.stock', compact('products', 'totalValue', 'lowStockCount', 'warehouseId'));
    }

    public function exportSales(Request $request)
    {
        // Apply same filters as sales() method
        $query = Invoice::query();

        // Include all statuses except draft
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        } else {
            $query->whereIn('status', ['paid', 'partially_paid', 'final', 'returned']);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        } else {
            $query->whereDate('created_at', '>=', now()->startOfMonth());
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        } else {
            $query->whereDate('created_at', '<=', now()->endOfMonth());
        }

        $invoices = $query->with(['customer', 'user', 'items.product'])->latest()->get();

        $filename = 'sales_report_' . date('Y-m-d') . '.xlsx';
        return Excel::download(new \App\Exports\SalesReportExport($invoices), $filename);
    }

    public function exportSalesPdf(Request $request)
    {
        // Apply same filters as sales() method
        $query = Invoice::query();

        // Include all statuses except draft
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        } else {
            $query->whereIn('status', ['paid', 'partially_paid', 'final', 'returned']);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        } else {
            $query->whereDate('created_at', '>=', now()->startOfMonth());
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        } else {
            $query->whereDate('created_at', '<=', now()->endOfMonth());
        }

        $invoices = $query->with(['customer', 'user', 'items.product'])->latest()->get();

        // Calculate totals (excluding returned invoices)
        $calculationQuery = Invoice::query();
        
        if ($request->filled('status') && $request->status !== 'returned') {
            $calculationQuery->where('status', $request->status);
        } else {
            $calculationQuery->whereIn('status', ['paid', 'partially_paid', 'final']);
        }
        
        $calculationQuery->where('status', '!=', 'returned');
        
        if ($request->filled('date_from')) {
            $calculationQuery->whereDate('created_at', '>=', $request->date_from);
        } else {
            $calculationQuery->whereDate('created_at', '>=', now()->startOfMonth());
        }

        if ($request->filled('date_to')) {
            $calculationQuery->whereDate('created_at', '<=', $request->date_to);
        } else {
            $calculationQuery->whereDate('created_at', '<=', now()->endOfMonth());
        }
        
        $allInvoices = $calculationQuery->with(['customer', 'user', 'items.product'])->latest()->get();
        
        // Calculate profit for each invoice
        foreach ($allInvoices as $invoice) {
            $invoice->profit = $invoice->items->sum(function ($item) {
                $revenue = $item->total;
                $cost = $item->quantity * ($item->product->purchase_price ?? 0);
                return $revenue - $cost;
            });
        }
        
        $totalSales = $allInvoices->sum('total');
        $totalPaid = $allInvoices->sum('paid_amount');
        $totalDue = $allInvoices->sum('due_amount');
        $totalProfit = $allInvoices->sum('profit');

        // Calculate profit for displayed invoices
        foreach ($invoices as $invoice) {
            $invoice->profit = $invoice->items->sum(function ($item) {
                $revenue = $item->total;
                $cost = $item->quantity * ($item->product->purchase_price ?? 0);
                return $revenue - $cost;
            });
        }

        // استخدام صفحة الطباعة لجميع اللغات
        return view('reports.sales.print', compact('invoices', 'totalSales', 'totalPaid', 'totalDue', 'totalProfit', 'request'));
    }

    // ========== FINANCIAL REPORTS ==========
    
    public function cashFlow(Request $request)
    {
        $dateFrom = $request->get('date_from', now()->startOfMonth()->format('Y-m-d'));
        $dateTo = $request->get('date_to', now()->endOfMonth()->format('Y-m-d'));

        // Cash Inflows (Sales Payments)
        $cashInflows = Payment::whereBetween('created_at', [$dateFrom, $dateTo])
            ->selectRaw('DATE(created_at) as date, SUM(amount) as total')
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        // Cash Outflows (Supplier Payments)
        $cashOutflows = SupplierPayment::whereBetween('payment_date', [$dateFrom, $dateTo])
            ->selectRaw('DATE(payment_date) as date, SUM(amount_paid) as total')
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        // Combine and calculate net cash flow
        $allDates = collect($cashInflows->pluck('date'))
            ->merge($cashOutflows->pluck('date'))
            ->unique()
            ->sort()
            ->values();

        $cashFlow = $allDates->map(function ($date) use ($cashInflows, $cashOutflows) {
            $inflow = $cashInflows->where('date', $date)->first()->total ?? 0;
            $outflow = $cashOutflows->where('date', $date)->first()->total ?? 0;
            
            return [
                'date' => $date,
                'inflow' => $inflow,
                'outflow' => $outflow,
                'net' => $inflow - $outflow,
            ];
        });

        $totalInflow = $cashInflows->sum('total');
        $totalOutflow = $cashOutflows->sum('total');
        $netCashFlow = $totalInflow - $totalOutflow;

        return view('reports.cash-flow', compact('cashFlow', 'totalInflow', 'totalOutflow', 'netCashFlow', 'dateFrom', 'dateTo'));
    }

    public function cashFlowPrint(Request $request)
    {
        $dateFrom = $request->get('date_from', now()->startOfMonth()->format('Y-m-d'));
        $dateTo = $request->get('date_to', now()->endOfMonth()->format('Y-m-d'));

        // Cash Inflows (Sales Payments)
        $cashInflows = Payment::whereBetween('created_at', [$dateFrom, $dateTo])
            ->selectRaw('DATE(created_at) as date, SUM(amount) as total')
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        // Cash Outflows (Supplier Payments)
        $cashOutflows = SupplierPayment::whereBetween('payment_date', [$dateFrom, $dateTo])
            ->selectRaw('DATE(payment_date) as date, SUM(amount_paid) as total')
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        // Combine and calculate net cash flow
        $allDates = collect($cashInflows->pluck('date'))
            ->merge($cashOutflows->pluck('date'))
            ->unique()
            ->sort()
            ->values();

        $cashFlow = $allDates->map(function ($date) use ($cashInflows, $cashOutflows) {
            $inflow = $cashInflows->where('date', $date)->first()->total ?? 0;
            $outflow = $cashOutflows->where('date', $date)->first()->total ?? 0;
            
            return [
                'date' => $date,
                'inflow' => $inflow,
                'outflow' => $outflow,
                'net' => $inflow - $outflow,
            ];
        });

        $totalInflow = $cashInflows->sum('total');
        $totalOutflow = $cashOutflows->sum('total');
        $netCashFlow = $totalInflow - $totalOutflow;

        return view('reports.cash-flow.print', compact('cashFlow', 'totalInflow', 'totalOutflow', 'netCashFlow', 'dateFrom', 'dateTo'));
    }

    public function profitLoss(Request $request)
    {
        $dateFrom = $request->get('date_from', now()->startOfMonth()->format('Y-m-d'));
        $dateTo = $request->get('date_to', now()->endOfMonth()->format('Y-m-d'));

        // Revenue (Sales)
        $revenue = Invoice::whereIn('status', ['paid', 'partially_paid', 'final'])
            ->whereBetween('created_at', [$dateFrom, $dateTo])
            ->sum('total');

        // Cost of Goods Sold (COGS)
        $cogs = InvoiceItem::whereHas('invoice', function ($q) use ($dateFrom, $dateTo) {
                $q->whereIn('status', ['paid', 'partially_paid', 'final'])
                  ->whereBetween('created_at', [$dateFrom, $dateTo]);
            })
            ->get()
            ->sum(function ($item) {
                return $item->quantity * ($item->product->purchase_price ?? 0);
            });

        // Gross Profit
        $grossProfit = $revenue - $cogs;

        // Operating Expenses (Purchase Invoices)
        $operatingExpenses = PurchaseInvoice::whereIn('status', ['confirmed', 'paid', 'partially_paid'])
            ->whereBetween('purchase_date', [$dateFrom, $dateTo])
            ->sum('total_amount');

        // Net Profit
        $netProfit = $grossProfit - $operatingExpenses;

        // Profit Margin
        $grossProfitMargin = $revenue > 0 ? ($grossProfit / $revenue) * 100 : 0;
        $netProfitMargin = $revenue > 0 ? ($netProfit / $revenue) * 100 : 0;

        return view('reports.profit-loss', compact(
            'revenue', 'cogs', 'grossProfit', 'operatingExpenses', 
            'netProfit', 'grossProfitMargin', 'netProfitMargin', 'dateFrom', 'dateTo'
        ));
    }

    public function profitLossPrint(Request $request)
    {
        $dateFrom = $request->get('date_from', now()->startOfMonth()->format('Y-m-d'));
        $dateTo = $request->get('date_to', now()->endOfMonth()->format('Y-m-d'));

        // Revenue (Sales)
        $revenue = Invoice::whereIn('status', ['paid', 'partially_paid', 'final'])
            ->whereBetween('created_at', [$dateFrom, $dateTo])
            ->sum('total');

        // Cost of Goods Sold (COGS)
        $cogs = InvoiceItem::whereHas('invoice', function ($q) use ($dateFrom, $dateTo) {
                $q->whereIn('status', ['paid', 'partially_paid', 'final'])
                  ->whereBetween('created_at', [$dateFrom, $dateTo]);
            })
            ->get()
            ->sum(function ($item) {
                return $item->quantity * ($item->product->purchase_price ?? 0);
            });

        // Gross Profit
        $grossProfit = $revenue - $cogs;

        // Operating Expenses (Purchase Invoices)
        $operatingExpenses = PurchaseInvoice::whereIn('status', ['confirmed', 'paid', 'partially_paid'])
            ->whereBetween('purchase_date', [$dateFrom, $dateTo])
            ->sum('total_amount');

        // Net Profit
        $netProfit = $grossProfit - $operatingExpenses;

        // Profit Margin
        $grossProfitMargin = $revenue > 0 ? ($grossProfit / $revenue) * 100 : 0;
        $netProfitMargin = $revenue > 0 ? ($netProfit / $revenue) * 100 : 0;

        return view('reports.profit-loss.print', compact(
            'revenue', 'cogs', 'grossProfit', 'operatingExpenses', 
            'netProfit', 'grossProfitMargin', 'netProfitMargin', 'dateFrom', 'dateTo'
        ));
    }

    // ========== SALES REPORTS (PERIODIC) ==========
    
    public function salesPeriodic(Request $request)
    {
        $period = $request->get('period', 'monthly'); // daily, weekly, monthly, yearly
        $dateFrom = $request->get('date_from');
        $dateTo = $request->get('date_to');

        $query = Invoice::whereIn('status', ['paid', 'partially_paid', 'final']);

        if ($dateFrom) {
            $query->whereDate('created_at', '>=', $dateFrom);
        }
        if ($dateTo) {
            $query->whereDate('created_at', '<=', $dateTo);
        }

        $invoices = $query->with(['customer', 'items.product'])->get();

        $periodicSales = collect();

        switch ($period) {
            case 'daily':
                $periodicSales = $invoices->groupBy(function ($invoice) {
                    return $invoice->created_at->format('Y-m-d');
                })->map(function ($group, $date) {
                    return [
                        'period' => $date,
                        'count' => $group->count(),
                        'total' => $group->sum('total'),
                        'profit' => $group->sum(function ($inv) {
                            return $inv->items->sum(function ($item) {
                                return $item->total - ($item->quantity * ($item->product->purchase_price ?? 0));
                            });
                        }),
                    ];
                })->sortBy('period')->values();
                break;

            case 'weekly':
                $periodicSales = $invoices->groupBy(function ($invoice) {
                    return $invoice->created_at->format('Y-W');
                })->map(function ($group, $week) {
                    return [
                        'period' => 'Week ' . substr($week, 5),
                        'count' => $group->count(),
                        'total' => $group->sum('total'),
                        'profit' => $group->sum(function ($inv) {
                            return $inv->items->sum(function ($item) {
                                return $item->total - ($item->quantity * ($item->product->purchase_price ?? 0));
                            });
                        }),
                    ];
                })->sortBy('period')->values();
                break;

            case 'monthly':
                $periodicSales = $invoices->groupBy(function ($invoice) {
                    return $invoice->created_at->format('Y-m');
                })->map(function ($group, $month) {
                    return [
                        'period' => $month,
                        'count' => $group->count(),
                        'total' => $group->sum('total'),
                        'profit' => $group->sum(function ($inv) {
                            return $inv->items->sum(function ($item) {
                                return $item->total - ($item->quantity * ($item->product->purchase_price ?? 0));
                            });
                        }),
                    ];
                })->sortBy('period')->values();
                break;

            case 'yearly':
                $periodicSales = $invoices->groupBy(function ($invoice) {
                    return $invoice->created_at->format('Y');
                })->map(function ($group, $year) {
                    return [
                        'period' => $year,
                        'count' => $group->count(),
                        'total' => $group->sum('total'),
                        'profit' => $group->sum(function ($inv) {
                            return $inv->items->sum(function ($item) {
                                return $item->total - ($item->quantity * ($item->product->purchase_price ?? 0));
                            });
                        }),
                    ];
                })->sortBy('period')->values();
                break;
        }

        $totalSales = $periodicSales->sum('total');
        $totalProfit = $periodicSales->sum('profit');
        $totalCount = $periodicSales->sum('count');

        return view('reports.sales-periodic', compact('periodicSales', 'period', 'totalSales', 'totalProfit', 'totalCount', 'dateFrom', 'dateTo'));
    }

    // ========== SUPPLIER REPORTS ==========
    
    public function suppliers(Request $request)
    {
        $type = $request->get('type', 'best'); // best or worst
        $dateFrom = $request->get('date_from');
        $dateTo = $request->get('date_to');

        $query = PurchaseInvoice::whereIn('status', ['confirmed', 'paid', 'partially_paid']);

        if ($dateFrom) {
            $query->whereDate('purchase_date', '>=', $dateFrom);
        }
        if ($dateTo) {
            $query->whereDate('purchase_date', '<=', $dateTo);
        }

        $purchaseInvoices = $query->with(['supplier', 'items'])->get();

        $supplierStats = $purchaseInvoices->groupBy('supplier_id')->map(function ($group) {
            $supplier = $group->first()->supplier;
            return [
                'supplier' => $supplier,
                'invoice_count' => $group->count(),
                'total_amount' => $group->sum('total_amount'),
                'amount_paid' => $group->sum('amount_paid'),
                'amount_remaining' => $group->sum('amount_remaining'),
                'items_count' => $group->sum(function ($inv) {
                    return $inv->items->sum('quantity');
                }),
            ];
        });

        if ($type === 'best') {
            $supplierStats = $supplierStats->sortByDesc('total_amount');
        } else {
            $supplierStats = $supplierStats->sortBy('total_amount');
        }

        $itemsPerPage = Setting::getItemsPerPage(20);
        $currentPage = $request->get('page', 1);
        $offset = ($currentPage - 1) * $itemsPerPage;
        $paginatedStats = $supplierStats->slice($offset, $itemsPerPage)->values();
        $supplierStatsPaginator = new \Illuminate\Pagination\LengthAwarePaginator(
            $paginatedStats,
            $supplierStats->count(),
            $itemsPerPage,
            $currentPage,
            ['path' => $request->url(), 'query' => $request->query()]
        );

        return view('reports.suppliers', compact('supplierStatsPaginator', 'type', 'dateFrom', 'dateTo'));
    }

    // ========== CUSTOMER REPORTS ==========
    
    public function customers(Request $request)
    {
        $type = $request->get('type', 'best'); // best or worst
        $dateFrom = $request->get('date_from');
        $dateTo = $request->get('date_to');

        $query = Invoice::whereIn('status', ['paid', 'partially_paid', 'final']);

        if ($dateFrom) {
            $query->whereDate('created_at', '>=', $dateFrom);
        }
        if ($dateTo) {
            $query->whereDate('created_at', '<=', $dateTo);
        }

        $invoices = $query->with(['customer', 'items'])->get();

        $customerStats = $invoices->groupBy('customer_id')->map(function ($group) {
            $customer = $group->first()->customer;
            return [
                'customer' => $customer,
                'invoice_count' => $group->count(),
                'total_amount' => $group->sum('total'),
                'paid_amount' => $group->sum('paid_amount'),
                'due_amount' => $group->sum('due_amount'),
                'items_count' => $group->sum(function ($inv) {
                    return $inv->items->sum('quantity');
                }),
            ];
        })->filter(function ($stat) {
            return $stat['customer'] !== null;
        });

        if ($type === 'best') {
            $customerStats = $customerStats->sortByDesc('total_amount');
        } else {
            $customerStats = $customerStats->sortBy('total_amount');
        }

        $itemsPerPage = Setting::getItemsPerPage(20);
        $currentPage = $request->get('page', 1);
        $offset = ($currentPage - 1) * $itemsPerPage;
        $paginatedStats = $customerStats->slice($offset, $itemsPerPage)->values();
        $customerStatsPaginator = new \Illuminate\Pagination\LengthAwarePaginator(
            $paginatedStats,
            $customerStats->count(),
            $itemsPerPage,
            $currentPage,
            ['path' => $request->url(), 'query' => $request->query()]
        );

        return view('reports.customers', compact('customerStatsPaginator', 'type', 'dateFrom', 'dateTo'));
    }

    // ========== PRODUCT MOVEMENT REPORT ==========
    
    public function productMovement(Request $request)
    {
        $productId = $request->get('product_id');
        $dateFrom = $request->get('date_from');
        $dateTo = $request->get('date_to');
        $type = $request->get('type', 'all'); // all, in, out

        $query = StockMovement::with(['product', 'user']);

        if ($productId) {
            $query->where('product_id', $productId);
        }

        if ($dateFrom) {
            $query->whereDate('created_at', '>=', $dateFrom);
        }
        if ($dateTo) {
            $query->whereDate('created_at', '<=', $dateTo);
        }

        if ($type === 'in') {
            $query->where('type', 'IN');
        } elseif ($type === 'out') {
            $query->where('type', 'OUT');
        }

        $itemsPerPage = Setting::getItemsPerPage(20);
        $movements = $query->latest()->paginate($itemsPerPage)->appends($request->query());

        $products = Product::orderBy('name')->get();

        return view('reports.product-movement', compact('movements', 'products', 'productId', 'dateFrom', 'dateTo', 'type'));
    }

    // ========== STAGNANT PRODUCTS REPORT ==========
    
    public function stagnantProducts(Request $request)
    {
        $days = $request->get('days', 90); // Products not sold in last X days
        $dateFrom = $request->get('date_from');
        $dateTo = $request->get('date_to');

        // Get all products
        $allProducts = Product::with('category')->get();

        // Get sold products in the period
        $query = InvoiceItem::whereHas('invoice', function ($q) {
                $q->whereIn('status', ['paid', 'partially_paid', 'final']);
            });

        if ($dateFrom) {
            $query->whereHas('invoice', function ($q) use ($dateFrom) {
                $q->whereDate('created_at', '>=', $dateFrom);
            });
        } else {
            $query->whereHas('invoice', function ($q) use ($days) {
                $q->whereDate('created_at', '>=', now()->subDays($days));
            });
        }

        if ($dateTo) {
            $query->whereHas('invoice', function ($q) use ($dateTo) {
                $q->whereDate('created_at', '<=', $dateTo);
            });
        }

        $soldProductIds = $query->pluck('product_id')->unique();

        // Get stagnant products (not sold in the period)
        $stagnantProducts = $allProducts->whereNotIn('id', $soldProductIds);

        // Get warehouse ID if multiple warehouses mode
        $warehouseId = null;
        if (StockManager::isMultipleWarehouses()) {
            $warehouseId = $request->get('warehouse_id');
        }
        
        // Calculate total value
        $totalValue = $stagnantProducts->sum(function ($product) use ($warehouseId) {
            return StockManager::getStockValue($product, $warehouseId);
        });

        // Paginate
        $itemsPerPage = Setting::getItemsPerPage(20);
        $currentPage = $request->get('page', 1);
        $offset = ($currentPage - 1) * $itemsPerPage;
        $paginatedProducts = $stagnantProducts->slice($offset, $itemsPerPage)->values();
        $stagnantProductsPaginator = new \Illuminate\Pagination\LengthAwarePaginator(
            $paginatedProducts,
            $stagnantProducts->count(),
            $itemsPerPage,
            $currentPage,
            ['path' => $request->url(), 'query' => $request->query()]
        );

        return view('reports.stagnant-products', compact('stagnantProductsPaginator', 'days', 'totalValue', 'dateFrom', 'dateTo'));
    }

    // ========== BALANCE SHEET REPORT ==========
    
    public function balanceSheet(Request $request)
    {
        $asOfDate = $request->get('as_of_date', now()->format('Y-m-d'));

        // ASSETS
        // 1. Cash (from Treasury)
        $totalIncome = Treasury::where('date', '<=', $asOfDate)
            ->where('type', 'income')
            ->sum('amount');
        $totalExpense = Treasury::where('date', '<=', $asOfDate)
            ->where('type', 'expense')
            ->sum('amount');
        $cashBalance = $totalIncome - $totalExpense;

        // 2. Inventory Value
        if (StockManager::isMultipleWarehouses()) {
            // Sum from all warehouses
            $inventoryValue = Product::with('warehouses')->get()->sum(function ($product) {
                return StockManager::getTotalStock($product) * ($product->purchase_price ?? 0);
            });
        } else {
            $inventoryValue = Product::sum(DB::raw('stock_quantity * purchase_price'));
        }

        // 3. Accounts Receivable (Customer balances)
        $accountsReceivable = Customer::get()->sum(function ($customer) {
            return $customer->calculateCurrentBalance();
        });

        // 4. Total Assets
        $totalAssets = $cashBalance + $inventoryValue + $accountsReceivable;

        // LIABILITIES
        // 1. Accounts Payable (Supplier balances)
        $accountsPayable = Supplier::get()->sum(function ($supplier) {
            return $supplier->calculateCurrentBalance();
        });

        // 2. Total Liabilities
        $totalLiabilities = $accountsPayable;

        // EQUITY
        $equity = $totalAssets - $totalLiabilities;

        // Additional breakdowns
        $assetsBreakdown = [
            'cash' => $cashBalance,
            'inventory' => $inventoryValue,
            'accounts_receivable' => $accountsReceivable,
        ];

        $liabilitiesBreakdown = [
            'accounts_payable' => $accountsPayable,
        ];

        return view('reports.balance-sheet', compact(
            'asOfDate',
            'assetsBreakdown',
            'liabilitiesBreakdown',
            'totalAssets',
            'totalLiabilities',
            'equity'
        ));
    }

    public function balanceSheetPrint(Request $request)
    {
        $asOfDate = $request->get('as_of_date', now()->format('Y-m-d'));

        // ASSETS
        // 1. Cash (from Treasury)
        $totalIncome = Treasury::where('date', '<=', $asOfDate)
            ->where('type', 'income')
            ->sum('amount');
        $totalExpense = Treasury::where('date', '<=', $asOfDate)
            ->where('type', 'expense')
            ->sum('amount');
        $cashBalance = $totalIncome - $totalExpense;

        // 2. Inventory Value
        if (StockManager::isMultipleWarehouses()) {
            // Sum from all warehouses
            $inventoryValue = Product::with('warehouses')->get()->sum(function ($product) {
                return StockManager::getTotalStock($product) * ($product->purchase_price ?? 0);
            });
        } else {
            $inventoryValue = Product::sum(DB::raw('stock_quantity * purchase_price'));
        }

        // 3. Accounts Receivable (Customer balances)
        $accountsReceivable = Customer::get()->sum(function ($customer) {
            return $customer->calculateCurrentBalance();
        });

        // 4. Total Assets
        $totalAssets = $cashBalance + $inventoryValue + $accountsReceivable;

        // LIABILITIES
        // 1. Accounts Payable (Supplier balances)
        $accountsPayable = Supplier::get()->sum(function ($supplier) {
            return $supplier->calculateCurrentBalance();
        });

        // 2. Total Liabilities
        $totalLiabilities = $accountsPayable;

        // EQUITY
        $equity = $totalAssets - $totalLiabilities;

        // Additional breakdowns
        $assetsBreakdown = [
            'cash' => $cashBalance,
            'inventory' => $inventoryValue,
            'accounts_receivable' => $accountsReceivable,
        ];

        $liabilitiesBreakdown = [
            'accounts_payable' => $accountsPayable,
        ];

        return view('reports.balance-sheet.print', compact(
            'asOfDate',
            'assetsBreakdown',
            'liabilitiesBreakdown',
            'totalAssets',
            'totalLiabilities',
            'equity'
        ));
    }

    // ========== COST ANALYSIS REPORT ==========
    
    public function costAnalysis(Request $request)
    {
        $dateFrom = $request->get('date_from', now()->startOfMonth()->format('Y-m-d'));
        $dateTo = $request->get('date_to', now()->endOfMonth()->format('Y-m-d'));
        $groupBy = $request->get('group_by', 'product'); // product, category, supplier

        // Get purchase invoices in the period
        $purchaseInvoices = PurchaseInvoice::whereIn('status', ['confirmed', 'paid', 'partially_paid'])
            ->whereBetween('purchase_date', [$dateFrom, $dateTo])
            ->with(['items.product.category', 'supplier'])
            ->get();

        // Get sales invoices for revenue comparison
        $salesInvoices = Invoice::whereIn('status', ['paid', 'partially_paid', 'final'])
            ->whereBetween('created_at', [$dateFrom, $dateTo])
            ->with(['items.product'])
            ->get();

        $totalRevenue = $salesInvoices->sum('total');
        $totalCost = $purchaseInvoices->sum('total_amount');
        $totalProfit = $totalRevenue - $totalCost;
        $profitMargin = $totalRevenue > 0 ? ($totalProfit / $totalRevenue) * 100 : 0;

        $costBreakdown = collect();

        switch ($groupBy) {
            case 'product':
                // Group by product
                $productCosts = [];
                foreach ($purchaseInvoices as $invoice) {
                    foreach ($invoice->items as $item) {
                        $productId = $item->product_id;
                        if (!isset($productCosts[$productId])) {
                            $productCosts[$productId] = [
                                'product' => $item->product,
                                'quantity' => 0,
                                'total_cost' => 0,
                                'average_cost' => 0,
                            ];
                        }
                        $productCosts[$productId]['quantity'] += $item->quantity;
                        $productCosts[$productId]['total_cost'] += $item->total;
                    }
                }
                foreach ($productCosts as $productId => $data) {
                    $data['average_cost'] = $data['quantity'] > 0 ? $data['total_cost'] / $data['quantity'] : 0;
                    if ($data['product']) {
                        $data['product']->load('category');
                    }
                    $costBreakdown->push($data);
                }
                $costBreakdown = $costBreakdown->sortByDesc('total_cost')->values();
                break;

            case 'category':
                // Group by category
                $categoryCosts = [];
                foreach ($purchaseInvoices as $invoice) {
                    foreach ($invoice->items as $item) {
                        $categoryId = $item->product->category_id ?? 0;
                        $categoryName = $item->product->category->name ?? trans('messages.uncategorized');
                        if (!isset($categoryCosts[$categoryId])) {
                            $categoryCosts[$categoryId] = [
                                'category' => $categoryName,
                                'quantity' => 0,
                                'total_cost' => 0,
                                'item_count' => 0,
                            ];
                        }
                        $categoryCosts[$categoryId]['quantity'] += $item->quantity;
                        $categoryCosts[$categoryId]['total_cost'] += $item->total;
                        $categoryCosts[$categoryId]['item_count']++;
                    }
                }
                foreach ($categoryCosts as $categoryId => $data) {
                    $costBreakdown->push($data);
                }
                $costBreakdown = $costBreakdown->sortByDesc('total_cost')->values();
                break;

            case 'supplier':
                // Group by supplier
                $supplierCosts = [];
                foreach ($purchaseInvoices as $invoice) {
                    $supplierId = $invoice->supplier_id ?? 0;
                    $supplierName = $invoice->supplier->name ?? trans('messages.unknown');
                    if (!isset($supplierCosts[$supplierId])) {
                        $supplierCosts[$supplierId] = [
                            'supplier' => $supplierName,
                            'invoice_count' => 0,
                            'total_cost' => 0,
                            'quantity' => 0,
                        ];
                    }
                    $supplierCosts[$supplierId]['invoice_count']++;
                    $supplierCosts[$supplierId]['total_cost'] += $invoice->total_amount;
                    $supplierCosts[$supplierId]['quantity'] += $invoice->items->sum('quantity');
                }
                foreach ($supplierCosts as $supplierId => $data) {
                    $costBreakdown->push($data);
                }
                $costBreakdown = $costBreakdown->sortByDesc('total_cost')->values();
                break;
        }

        // Paginate
        $itemsPerPage = Setting::getItemsPerPage(20);
        $currentPage = $request->get('page', 1);
        $offset = ($currentPage - 1) * $itemsPerPage;
        $paginatedBreakdown = $costBreakdown->slice($offset, $itemsPerPage)->values();
        $costBreakdownPaginator = new \Illuminate\Pagination\LengthAwarePaginator(
            $paginatedBreakdown,
            $costBreakdown->count(),
            $itemsPerPage,
            $currentPage,
            ['path' => $request->url(), 'query' => $request->query()]
        );

        return view('reports.cost-analysis', compact(
            'costBreakdownPaginator',
            'groupBy',
            'dateFrom',
            'dateTo',
            'totalRevenue',
            'totalCost',
            'totalProfit',
            'profitMargin'
        ));
    }

    public function costAnalysisPrint(Request $request)
    {
        $dateFrom = $request->get('date_from', now()->startOfMonth()->format('Y-m-d'));
        $dateTo = $request->get('date_to', now()->endOfMonth()->format('Y-m-d'));
        $groupBy = $request->get('group_by', 'product'); // product, category, supplier

        // Get purchase invoices in the period
        $purchaseInvoices = PurchaseInvoice::whereIn('status', ['confirmed', 'paid', 'partially_paid'])
            ->whereBetween('purchase_date', [$dateFrom, $dateTo])
            ->with(['items.product.category', 'supplier'])
            ->get();

        // Get sales invoices for revenue comparison
        $salesInvoices = Invoice::whereIn('status', ['paid', 'partially_paid', 'final'])
            ->whereBetween('created_at', [$dateFrom, $dateTo])
            ->with(['items.product'])
            ->get();

        $totalRevenue = $salesInvoices->sum('total');
        $totalCost = $purchaseInvoices->sum('total_amount');
        $totalProfit = $totalRevenue - $totalCost;
        $profitMargin = $totalRevenue > 0 ? ($totalProfit / $totalRevenue) * 100 : 0;

        $costBreakdown = collect();

        switch ($groupBy) {
            case 'product':
                // Group by product
                $productCosts = [];
                foreach ($purchaseInvoices as $invoice) {
                    foreach ($invoice->items as $item) {
                        $productId = $item->product_id;
                        if (!isset($productCosts[$productId])) {
                            $productCosts[$productId] = [
                                'product' => $item->product,
                                'quantity' => 0,
                                'total_cost' => 0,
                                'average_cost' => 0,
                            ];
                        }
                        $productCosts[$productId]['quantity'] += $item->quantity;
                        $productCosts[$productId]['total_cost'] += $item->total;
                    }
                }
                foreach ($productCosts as $productId => $data) {
                    $data['average_cost'] = $data['quantity'] > 0 ? $data['total_cost'] / $data['quantity'] : 0;
                    if ($data['product']) {
                        $data['product']->load('category');
                    }
                    $costBreakdown->push($data);
                }
                $costBreakdown = $costBreakdown->sortByDesc('total_cost')->values();
                break;

            case 'category':
                // Group by category
                $categoryCosts = [];
                foreach ($purchaseInvoices as $invoice) {
                    foreach ($invoice->items as $item) {
                        $categoryId = $item->product->category_id ?? 0;
                        $categoryName = $item->product->category->name ?? trans('messages.uncategorized');
                        if (!isset($categoryCosts[$categoryId])) {
                            $categoryCosts[$categoryId] = [
                                'category' => $categoryName,
                                'quantity' => 0,
                                'total_cost' => 0,
                                'item_count' => 0,
                            ];
                        }
                        $categoryCosts[$categoryId]['quantity'] += $item->quantity;
                        $categoryCosts[$categoryId]['total_cost'] += $item->total;
                        $categoryCosts[$categoryId]['item_count']++;
                    }
                }
                foreach ($categoryCosts as $categoryId => $data) {
                    $costBreakdown->push($data);
                }
                $costBreakdown = $costBreakdown->sortByDesc('total_cost')->values();
                break;

            case 'supplier':
                // Group by supplier
                $supplierCosts = [];
                foreach ($purchaseInvoices as $invoice) {
                    $supplierId = $invoice->supplier_id ?? 0;
                    $supplierName = $invoice->supplier->name ?? trans('messages.unknown');
                    if (!isset($supplierCosts[$supplierId])) {
                        $supplierCosts[$supplierId] = [
                            'supplier' => $supplierName,
                            'invoice_count' => 0,
                            'total_cost' => 0,
                            'quantity' => 0,
                        ];
                    }
                    $supplierCosts[$supplierId]['invoice_count']++;
                    $supplierCosts[$supplierId]['total_cost'] += $invoice->total_amount;
                    $supplierCosts[$supplierId]['quantity'] += $invoice->items->sum('quantity');
                }
                foreach ($supplierCosts as $supplierId => $data) {
                    $costBreakdown->push($data);
                }
                $costBreakdown = $costBreakdown->sortByDesc('total_cost')->values();
                break;
        }

        return view('reports.cost-analysis.print', compact(
            'costBreakdown',
            'groupBy',
            'dateFrom',
            'dateTo',
            'totalRevenue',
            'totalCost',
            'totalProfit',
            'profitMargin'
        ));
    }

    public function warehouses(Request $request)
    {
        // Check if multiple warehouses mode is enabled
        if (!StockManager::isMultipleWarehouses()) {
            return redirect()->route('reports.index')
                ->with('error', trans('messages.multiple_warehouses_not_enabled'));
        }

        $query = Warehouse::withCount('products');

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('name_ar', 'like', "%{$search}%")
                  ->orWhere('code', 'like', "%{$search}%");
            });
        }

        if ($request->filled('is_active')) {
            $query->where('is_active', $request->is_active === '1');
        }

        // Get all warehouses for statistics
        $allWarehouses = Warehouse::active()->get();
        
        // Calculate statistics
        $totalWarehouses = $allWarehouses->count();
        $totalStockValue = $allWarehouses->sum('total_stock_value');
        $totalProducts = $allWarehouses->sum('total_products_count');
        $activeWarehouses = $allWarehouses->where('is_active', true)->count();
        $defaultWarehouse = $allWarehouses->where('is_default', true)->first();

        // Get warehouse details with products
        $warehouses = $query->get()->map(function ($warehouse) {
            $warehouse->load(['products' => function ($query) {
                $query->where('product_warehouse.quantity', '>', 0)
                      ->orderBy('product_warehouse.quantity', 'desc')
                      ->limit(10);
            }]);
            return $warehouse;
        });

        // Paginate warehouses
        $itemsPerPage = Setting::getItemsPerPage(20);
        $currentPage = $request->get('page', 1);
        $offset = ($currentPage - 1) * $itemsPerPage;
        $paginatedWarehouses = $warehouses->slice($offset, $itemsPerPage)->values();
        $warehousesPaginator = new \Illuminate\Pagination\LengthAwarePaginator(
            $paginatedWarehouses,
            $warehouses->count(),
            $itemsPerPage,
            $currentPage,
            ['path' => $request->url(), 'query' => $request->query()]
        );

        return view('reports.warehouses', compact(
            'warehousesPaginator',
            'totalWarehouses',
            'totalStockValue',
            'totalProducts',
            'activeWarehouses',
            'defaultWarehouse'
        ));
    }
}

