<?php

namespace App\Http\Controllers;

use App\Models\StockTake;
use App\Models\StockTakeItem;
use App\Models\Product;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class StockTakeController extends Controller
{
    public function index(Request $request)
    {
        $query = StockTake::with(['creator', 'completer']);

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where('reference_number', 'like', "%{$search}%");
        }

        $itemsPerPage = Setting::getItemsPerPage(20);
        $stockTakes = $query->latest()->paginate($itemsPerPage)->appends($request->query());

        return view('stock-takes.index', compact('stockTakes'));
    }

    public function create()
    {
        $products = Product::all();
        return view('stock-takes.create', compact('products'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'stock_take_date' => 'required|date',
            'type' => 'required|in:full,partial',
            'notes' => 'nullable|string',
            'products' => 'required|array|min:1',
            'products.*' => 'exists:products,id',
        ]);

        DB::beginTransaction();
        try {
            $stockTake = StockTake::create([
                'stock_take_date' => $validated['stock_take_date'],
                'status' => 'draft',
                'type' => $validated['type'],
                'notes' => $validated['notes'] ?? null,
                'created_by' => auth()->id(),
            ]);

            foreach ($validated['products'] as $productId) {
                $product = Product::find($productId);
                StockTakeItem::create([
                    'stock_take_id' => $stockTake->id,
                    'product_id' => $productId,
                    'system_quantity' => $product->stock_quantity,
                    'counted_quantity' => $product->stock_quantity, // Initially same
                    'unit_cost' => $product->purchase_price,
                ]);
            }

            DB::commit();
            return redirect()->route('stock-takes.show', $stockTake)
                ->with('success', trans('messages.stock_take_created'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.error_occurred') . ': ' . $e->getMessage());
        }
    }

    public function show(StockTake $stockTake)
    {
        $stockTake->load(['items.product', 'creator', 'completer']);
        return view('stock-takes.show', compact('stockTake'));
    }

    public function edit(StockTake $stockTake)
    {
        if (!$stockTake->isDraft()) {
            return back()->with('error', trans('messages.cannot_edit_non_draft_stock_take'));
        }

        $stockTake->load('items.product');
        return view('stock-takes.edit', compact('stockTake'));
    }

    public function update(Request $request, StockTake $stockTake)
    {
        if (!$stockTake->isDraft()) {
            return back()->with('error', trans('messages.cannot_edit_non_draft_stock_take'));
        }

        $validated = $request->validate([
            'counted_quantities' => 'required|array',
            'counted_quantities.*' => 'required|integer|min:0',
        ]);

        DB::beginTransaction();
        try {
            foreach ($validated['counted_quantities'] as $itemId => $countedQuantity) {
                $item = StockTakeItem::find($itemId);
                if ($item && $item->stock_take_id == $stockTake->id) {
                    $item->update(['counted_quantity' => $countedQuantity]);
                }
            }

            DB::commit();
            return redirect()->route('stock-takes.show', $stockTake)
                ->with('success', trans('messages.stock_take_updated'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.error_occurred') . ': ' . $e->getMessage());
        }
    }

    public function complete(StockTake $stockTake)
    {
        if ($stockTake->isCompleted()) {
            return back()->with('error', trans('messages.stock_take_already_completed'));
        }

        DB::beginTransaction();
        try {
            $stockTake->complete(auth()->id());
            DB::commit();
            return back()->with('success', trans('messages.stock_take_completed'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.error_occurred') . ': ' . $e->getMessage());
        }
    }

    public function destroy(StockTake $stockTake)
    {
        if (!$stockTake->isDraft()) {
            return back()->with('error', trans('messages.cannot_delete_non_draft_stock_take'));
        }

        $stockTake->delete();
        return redirect()->route('stock-takes.index')
            ->with('success', trans('messages.stock_take_deleted'));
    }
}
