<?php

namespace App\Http\Controllers;

use App\Models\Setting;
use App\Models\Supplier;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Traits\Sortable;
use Maatwebsite\Excel\Facades\Excel;
use App\Imports\SuppliersImport;

class SupplierController extends Controller
{
    use Sortable;

    public function index(Request $request)
    {
        $query = Supplier::query();

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('phone', 'like', "%{$search}%");
            });
        }

        // Filter by suppliers with debts (balance > 0)
        if ($request->filled('filter') && $request->filter === 'with_debts') {
            // We'll filter after calculating balances
        }

        $allowedColumns = ['name', 'email', 'phone', 'created_at'];
        $sort = $request->get('sort', 'created_at');
        $direction = $request->get('direction', 'desc');

        if (in_array($sort, $allowedColumns)) {
            $query->orderBy($sort, $direction);
        } else {
            $query->latest();
        }

        $itemsPerPage = Setting::getItemsPerPage(20);
        $suppliers = $query->with('purchaseInvoices')
            ->paginate($itemsPerPage)
            ->appends($request->query());

        // Calculate current balance for each supplier
        $suppliers->getCollection()->transform(function ($supplier) {
            $supplier->calculated_balance = $supplier->calculateCurrentBalance();
            return $supplier;
        });

        // Filter by debts after calculating balances
        if ($request->filled('filter') && $request->filter === 'with_debts') {
            $suppliers->setCollection(
                $suppliers->getCollection()->filter(function ($supplier) {
                    return ($supplier->calculated_balance ?? $supplier->calculateCurrentBalance()) > 0;
                })
            );
        }

        return view('suppliers.index', compact('suppliers', 'sort', 'direction'));
    }

    public function printWithDebts(Request $request)
    {
        $suppliers = Supplier::with('purchaseInvoices')->get();
        
        // Calculate balance and filter suppliers with debts
        $suppliersWithDebts = $suppliers->map(function ($supplier) {
            $supplier->calculated_balance = $supplier->calculateCurrentBalance();
            return $supplier;
        })->filter(function ($supplier) {
            return $supplier->calculated_balance > 0;
        })->sortByDesc('calculated_balance');

        $totalDebts = $suppliersWithDebts->sum('calculated_balance');
        $storeName = Setting::get('store_name', 'Gestock Pro');
        $storeLogo = Setting::get('store_logo');
        $locale = app()->getLocale();
        $isRTL = $locale === 'ar';
        $dir = $isRTL ? 'rtl' : 'ltr';

        return view('suppliers.print-with-debts', compact('suppliersWithDebts', 'totalDebts', 'storeName', 'storeLogo', 'locale', 'isRTL', 'dir'));
    }

    public function create()
    {
        return view('suppliers.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'nullable|string|max:50',
            'email' => 'nullable|email|unique:suppliers,email',
            'address' => 'nullable|string',
            'notes' => 'nullable|string',
        ]);

        Supplier::create($validated);

        return redirect()->route('suppliers.index')
            ->with('success', trans('messages.supplier_created'));
    }

    public function edit(Supplier $supplier)
    {
        return view('suppliers.edit', compact('supplier'));
    }

    public function update(Request $request, Supplier $supplier)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'nullable|string|max:50',
            'email' => ['nullable', 'email', Rule::unique('suppliers')->ignore($supplier->id)],
            'address' => 'nullable|string',
            'notes' => 'nullable|string',
        ]);

        $supplier->update($validated);

        return redirect()->route('suppliers.index')
            ->with('success', trans('messages.supplier_updated'));
    }

    public function show(Supplier $supplier)
    {
        $supplier->load(['purchaseInvoices.items.product', 'purchaseInvoices.payments', 'cheques', 'promissoryNotes']);
        $supplier->calculated_balance = $supplier->calculateCurrentBalance();
        return view('suppliers.show', compact('supplier'));
    }

    public function destroy(Supplier $supplier)
    {
        // Check if supplier has purchase invoices
        if ($supplier->purchaseInvoices()->count() > 0) {
            return redirect()->route('suppliers.index')
                ->with('error', trans('messages.cannot_delete_suppliers_with_invoices'));
        }

        $supplier->delete();

        return redirect()->route('suppliers.index')
            ->with('success', trans('messages.supplier_deleted'));
    }

    public function bulkDelete(Request $request)
    {
        // Check permissions - Only Admin and Super Admin can bulk delete
        $user = auth()->user();
        if (!$user->hasAnyRole(['Admin', 'Super Admin'])) {
            abort(403, trans('messages.unauthorized_action'));
        }

        $request->validate([
            'selected_ids' => 'required|array|min:1',
            'selected_ids.*' => 'exists:suppliers,id',
        ]);

        try {
            $selectedIds = $request->selected_ids;
            $suppliers = Supplier::whereIn('id', $selectedIds)->get();
            
            // Check for suppliers with purchase invoices
            $suppliersWithInvoices = $suppliers->filter(function ($supplier) {
                return $supplier->purchaseInvoices()->count() > 0;
            });

            if ($suppliersWithInvoices->count() > 0) {
                return back()->with('error', trans('messages.cannot_delete_suppliers_with_invoices'));
            }
            
            // Log before deletion
            foreach ($suppliers as $supplier) {
                \App\Models\AuditLog::create([
                    'action' => 'bulk_delete',
                    'model_type' => Supplier::class,
                    'model_id' => $supplier->id,
                    'old_values' => $supplier->toArray(),
                    'new_values' => null,
                    'user_id' => auth()->id(),
                    'ip_address' => request()->ip(),
                    'user_agent' => request()->userAgent(),
                ]);
            }

            Supplier::whereIn('id', $selectedIds)->delete();

            return redirect()->route('suppliers.index')
                ->with('success', trans('messages.bulk_delete_success', ['count' => count($selectedIds)]));
        } catch (\Exception $e) {
            return back()->with('error', trans('messages.bulk_delete_error') . ': ' . $e->getMessage());
        }
    }

    public function importForm()
    {
        // Check permissions - Only Admin and Super Admin can import
        $user = auth()->user();
        if (!$user->hasAnyRole(['Admin', 'Super Admin'])) {
            abort(403, trans('messages.unauthorized_action'));
        }

        return view('suppliers.import');
    }

    public function import(Request $request)
    {
        // Check permissions
        $user = auth()->user();
        if (!$user->hasAnyRole(['Admin', 'Super Admin'])) {
            abort(403, trans('messages.unauthorized_action'));
        }

        $request->validate([
            'file' => 'required|file|mimes:xlsx,xls|max:10240',
        ]);

        try {
            $file = $request->file('file');
            
            if (!$file->isValid()) {
                return back()->with('error', trans('messages.import_error') . ': ' . 'Invalid file.');
            }

            $import = new SuppliersImport();
            
            Excel::import($import, $file);
            
            $imported = $import->getImported();
            $failed = $import->getFailed();
            $errors = $import->getErrors();

            if ($imported === 0 && $failed === 0) {
                return back()->with('error', trans('messages.excel_file_empty'));
            }

            $message = trans('messages.import_completed', [
                'imported' => $imported,
                'failed' => $failed,
            ]);

            if ($failed > 0 && !empty($errors)) {
                return back()->with('warning', $message)->with('import_errors', $errors);
            }

            return back()->with('success', $message);

        } catch (\Maatwebsite\Excel\Validators\ValidationException $e) {
            $failures = $e->failures();
            $errors = [];
            
            foreach ($failures as $failure) {
                $errors[] = [
                    'row' => $failure->row(),
                    'name' => $failure->values()['name'] ?? 'N/A',
                    'error' => implode(', ', $failure->errors()),
                ];
            }
            
            return back()->with('error', trans('messages.import_error'))
                        ->with('import_errors', $errors);
                        
        } catch (\Maatwebsite\Excel\Exceptions\SheetNotFoundException $e) {
            return back()->with('error', trans('messages.import_error') . ': ' . 'Sheet not found in Excel file.');
        } catch (\PhpOffice\PhpSpreadsheet\Reader\Exception $e) {
            return back()->with('error', trans('messages.import_error') . ': ' . 'Error reading Excel file. Please check if the file is valid.');
        } catch (\Exception $e) {
            \Log::error('Supplier import error: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            return back()->with('error', trans('messages.import_error') . ': ' . $e->getMessage());
        }
    }

    public function downloadTemplate()
    {
        $data = [
            ['name', 'phone', 'email', 'address', 'notes'],
            ['Supplier 1', '123456789', 'supplier1@example.com', 'Address 1', 'Notes 1'],
            ['Supplier 2', '987654321', 'supplier2@example.com', 'Address 2', 'Notes 2'],
        ];

        $filename = 'suppliers_import_template_' . date('Y-m-d') . '.xlsx';
        
        $export = new class($data) implements \Maatwebsite\Excel\Concerns\FromArray {
            protected $data;
            
            public function __construct($data) {
                $this->data = $data;
            }
            
            public function array(): array {
                return $this->data;
            }
        };
        
        return Excel::download($export, $filename);
    }
}

