<?php

namespace App\Http\Controllers;

use App\Models\PurchaseInvoice;
use App\Models\SupplierPayment;
use App\Models\Treasury;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class SupplierPaymentController extends Controller
{
    public function store(Request $request, PurchaseInvoice $purchaseInvoice)
    {
        $validated = $request->validate([
            'amount_paid' => 'required|numeric|min:0.01',
            'payment_method' => 'required|in:cash,bank_transfer,credit,cheque,online,other,promissory_note',
            'payment_reference' => 'nullable|string|max:255',
            'payment_date' => 'required|date',
            'notes' => 'nullable|string',
        ]);

        if ($purchaseInvoice->isDraft()) {
            return back()->with('error', trans('messages.cannot_add_payment_to_draft'));
        }

        if ($purchaseInvoice->isReturned()) {
            return back()->with('error', trans('messages.cannot_add_payment_to_returned'));
        }

        $remainingDue = $purchaseInvoice->amount_remaining;

        if ($validated['amount_paid'] > $remainingDue) {
            return back()->with('error', trans('messages.payment_exceeds_remaining') . ': ' . number_format($remainingDue, 2));
        }

        DB::beginTransaction();
        try {
            SupplierPayment::create([
                'purchase_invoice_id' => $purchaseInvoice->id,
                'amount_paid' => $validated['amount_paid'],
                'payment_method' => $validated['payment_method'],
                'payment_reference' => $validated['payment_reference'] ?? null,
                'payment_date' => $validated['payment_date'],
                'user_id' => auth()->id(),
                'notes' => $validated['notes'] ?? null,
            ]);

            $purchaseInvoice->updateTotals();

            // Create treasury entry for supplier payment
            $paymentMethod = $validated['payment_method'] === 'bank_transfer' ? 'bank' : (in_array($validated['payment_method'], ['cheque', 'credit', 'online', 'promissory_note']) ? 'other' : $validated['payment_method']);
            Treasury::create([
                'date' => $validated['payment_date'],
                'type' => 'expense',
                'reference_type' => 'purchase_invoice',
                'reference_id' => $purchaseInvoice->id,
                'amount' => $validated['amount_paid'],
                'description' => trans('messages.payment_to_supplier_for_purchase_invoice') . ' #' . $purchaseInvoice->invoice_number,
                'payment_method' => $paymentMethod,
                'created_by' => auth()->id(),
            ]);

            DB::commit();

            return redirect()->route('purchase-invoices.show', $purchaseInvoice)
                ->with('success', trans('messages.supplier_payment_recorded'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.error_occurred') . ': ' . $e->getMessage());
        }
    }

    public function destroy(SupplierPayment $supplierPayment)
    {
        $invoice = $supplierPayment->purchaseInvoice;

        DB::beginTransaction();
        try {
            $paymentAmount = $supplierPayment->amount_paid;
            $paymentMethod = $supplierPayment->payment_method === 'bank_transfer' ? 'bank' : (in_array($supplierPayment->payment_method, ['cheque', 'credit', 'online', 'promissory_note']) ? 'other' : $supplierPayment->payment_method);
            $supplierPayment->delete();
            $invoice->updateTotals();

            // Delete treasury entry for supplier payment
            Treasury::where('reference_type', 'purchase_invoice')
                ->where('reference_id', $invoice->id)
                ->where('amount', $paymentAmount)
                ->where('payment_method', $paymentMethod)
                ->where('type', 'expense')
                ->orderBy('created_at', 'desc')
                ->first()
                ?->delete();

            DB::commit();

            return redirect()->route('purchase-invoices.show', $invoice)
                ->with('success', trans('messages.supplier_payment_deleted'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.error_occurred') . ': ' . $e->getMessage());
        }
    }
}

