<?php

namespace App\Http\Controllers;

use App\Models\Setting;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;
use Spatie\Permission\Models\Role;
use App\Traits\Sortable;

class UserController extends Controller
{
    use Sortable;

    public function __construct()
    {
        // Only Admin and Super Admin can access user management
        $this->middleware(function ($request, $next) {
            if (!auth()->user()->hasAnyRole(['Admin', 'Super Admin'])) {
                abort(403, trans('messages.unauthorized_action'));
            }
            return $next($request);
        });
    }

    public function index(Request $request)
    {
        $query = User::with('roles');

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            });
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('role')) {
            $query->whereHas('roles', function ($q) use ($request) {
                $q->where('name', $request->role);
            });
        }

        $allowedColumns = ['name', 'email', 'created_at'];
        $sort = $request->get('sort', 'created_at');
        $direction = $request->get('direction', 'desc');

        if (in_array($sort, $allowedColumns)) {
            $query->orderBy($sort, $direction);
        } else {
            $query->latest();
        }

        $itemsPerPage = Setting::getItemsPerPage(20);
        $users = $query->paginate($itemsPerPage)->appends($request->query());
        $roles = Role::all();
        return view('users.index', compact('users', 'roles', 'sort', 'direction'));
    }

    public function create()
    {
        $roles = Role::all();
        return view('users.create', compact('roles'));
    }

    public function store(Request $request)
    {
        // Only Super Admin can create Admin users
        if ($request->role === 'Admin' && !auth()->user()->hasRole('Super Admin')) {
            return back()->withInput()->with('error', trans('messages.only_super_admin_can_create_admin'));
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|string|min:6|confirmed',
            'role' => 'required|exists:roles,name',
            'status' => 'nullable|in:active,inactive',
            'phone' => 'nullable|string|max:50',
            'avatar' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $avatarPath = null;
        if ($request->hasFile('avatar')) {
            $avatarPath = $request->file('avatar')->store('avatars', 'public');
        }

        $user = User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'password' => Hash::make($validated['password']),
            'status' => $validated['status'] ?? 'active',
            'phone' => $validated['phone'] ?? null,
            'avatar' => $avatarPath,
        ]);

        $user->assignRole($validated['role']);

        // Log creation in audit_logs
        \App\Models\AuditLog::create([
            'action' => 'create',
            'model_type' => User::class,
            'model_id' => $user->id,
            'old_values' => null,
            'new_values' => $user->toArray(),
            'user_id' => auth()->id(),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
        ]);

        return redirect()->route('users.index')
            ->with('success', trans('messages.user_created_successfully'));
    }

    public function edit(User $user)
    {
        $roles = Role::all();
        return view('users.edit', compact('user', 'roles'));
    }

    public function update(Request $request, User $user)
    {
        // Only Super Admin can edit Admin users
        if ($user->hasRole('Admin') && !auth()->user()->hasRole('Super Admin')) {
            return back()->withInput()->with('error', trans('messages.only_super_admin_can_edit_admin'));
        }

        // Only Super Admin can assign Admin role
        if ($request->role === 'Admin' && !auth()->user()->hasRole('Super Admin')) {
            return back()->withInput()->with('error', trans('messages.only_super_admin_can_assign_admin_role'));
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => ['required', 'email', Rule::unique('users')->ignore($user->id)],
            'password' => 'nullable|string|min:6|confirmed',
            'role' => 'required|exists:roles,name',
            'status' => 'nullable|in:active,inactive',
            'phone' => 'nullable|string|max:50',
            'avatar' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $oldValues = $user->toArray();

        // Handle avatar upload
        if ($request->hasFile('avatar')) {
            // Delete old avatar if exists
            if ($user->avatar) {
                Storage::disk('public')->delete($user->avatar);
            }
            $avatarPath = $request->file('avatar')->store('avatars', 'public');
            $validated['avatar'] = $avatarPath;
        }

        $user->update([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'status' => $validated['status'] ?? $user->status,
            'phone' => $validated['phone'] ?? $user->phone,
        ]);

        if (isset($validated['avatar'])) {
            $user->update(['avatar' => $validated['avatar']]);
        }

        if ($request->filled('password')) {
            $user->update(['password' => Hash::make($validated['password'])]);
        }

        $user->syncRoles([$validated['role']]);

        // Log update in audit_logs
        \App\Models\AuditLog::create([
            'action' => 'update',
            'model_type' => User::class,
            'model_id' => $user->id,
            'old_values' => $oldValues,
            'new_values' => $user->fresh()->toArray(),
            'user_id' => auth()->id(),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
        ]);

        return redirect()->route('users.index')
            ->with('success', trans('messages.user_updated_successfully'));
    }

    public function destroy(User $user)
    {
        // Cannot delete own account
        if ($user->id === auth()->id()) {
            return back()->with('error', trans('messages.cannot_delete_own_account'));
        }

        // Only Super Admin can delete Admin users
        if ($user->hasRole('Admin') && !auth()->user()->hasRole('Super Admin')) {
            return back()->with('error', trans('messages.only_super_admin_can_delete_admin'));
        }

        // Log deletion in audit_logs before deleting
        $userData = $user->toArray();
        \App\Models\AuditLog::create([
            'action' => 'delete',
            'model_type' => User::class,
            'model_id' => $user->id,
            'old_values' => $userData,
            'new_values' => null,
            'user_id' => auth()->id(),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
        ]);

        // Delete avatar if exists
        if ($user->avatar) {
            Storage::disk('public')->delete($user->avatar);
        }

        $user->delete();

        return redirect()->route('users.index')
            ->with('success', trans('messages.user_deleted_successfully'));
    }

    public function resetPassword(Request $request, User $user)
    {
        $validated = $request->validate([
            'password' => 'required|string|min:6|confirmed',
        ]);

        $user->update([
            'password' => Hash::make($validated['password']),
        ]);

        // Log password reset in audit_logs
        \App\Models\AuditLog::create([
            'action' => 'reset_password',
            'model_type' => User::class,
            'model_id' => $user->id,
            'old_values' => null,
            'new_values' => ['password_reset' => true],
            'user_id' => auth()->id(),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
        ]);

        return back()->with('success', trans('messages.password_reset_successfully'));
    }
}

