<?php

namespace App\Http\Controllers;

use App\Models\Warehouse;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Traits\Sortable;

class WarehouseController extends Controller
{
    use Sortable;

    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index(Request $request)
    {
        // Check if multiple warehouses mode is enabled
        if (Setting::get('warehouse_mode', 'single') !== 'multiple') {
            return redirect()->route('settings.index')
                ->with('error', trans('messages.multiple_warehouses_not_enabled'));
        }

        $query = Warehouse::withCount('products');

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('name_ar', 'like', "%{$search}%")
                  ->orWhere('code', 'like', "%{$search}%");
            });
        }

        if ($request->filled('is_active')) {
            $query->where('is_active', $request->is_active === '1');
        }

        $allowedColumns = ['name', 'code', 'is_default', 'is_active', 'created_at'];
        $sort = $request->get('sort', 'created_at');
        $direction = $request->get('direction', 'desc');

        if (in_array($sort, $allowedColumns)) {
            $query->orderBy($sort, $direction);
        } else {
            $query->latest();
        }

        $itemsPerPage = Setting::getItemsPerPage(20);
        $warehouses = $query->paginate($itemsPerPage)->appends($request->query());

        return view('warehouses.index', compact('warehouses', 'sort', 'direction'));
    }

    public function create()
    {
        if (Setting::get('warehouse_mode', 'single') !== 'multiple') {
            return redirect()->route('settings.index')
                ->with('error', trans('messages.multiple_warehouses_not_enabled'));
        }

        return view('warehouses.create');
    }

    public function store(Request $request)
    {
        if (Setting::get('warehouse_mode', 'single') !== 'multiple') {
            return redirect()->route('settings.index')
                ->with('error', trans('messages.multiple_warehouses_not_enabled'));
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'name_ar' => 'nullable|string|max:255',
            'code' => 'nullable|string|max:50|unique:warehouses,code',
            'address' => 'nullable|string',
            'phone' => 'nullable|string|max:50',
            'is_default' => 'nullable|boolean',
            'is_active' => 'nullable|boolean',
            'notes' => 'nullable|string',
        ]);

        // If this is the first warehouse, make it default
        if (Warehouse::count() === 0) {
            $validated['is_default'] = true;
        }

        Warehouse::create($validated);

        return redirect()->route('warehouses.index')
            ->with('success', trans('messages.warehouse_created'));
    }

    public function show(Warehouse $warehouse)
    {
        if (Setting::get('warehouse_mode', 'single') !== 'multiple') {
            return redirect()->route('settings.index')
                ->with('error', trans('messages.multiple_warehouses_not_enabled'));
        }

        $warehouse->load(['products' => function ($query) {
            $query->where('product_warehouse.quantity', '>', 0)
                  ->orderBy('product_warehouse.quantity', 'desc');
        }]);

        return view('warehouses.show', compact('warehouse'));
    }

    public function edit(Warehouse $warehouse)
    {
        if (Setting::get('warehouse_mode', 'single') !== 'multiple') {
            return redirect()->route('settings.index')
                ->with('error', trans('messages.multiple_warehouses_not_enabled'));
        }

        return view('warehouses.edit', compact('warehouse'));
    }

    public function update(Request $request, Warehouse $warehouse)
    {
        if (Setting::get('warehouse_mode', 'single') !== 'multiple') {
            return redirect()->route('settings.index')
                ->with('error', trans('messages.multiple_warehouses_not_enabled'));
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'name_ar' => 'nullable|string|max:255',
            'code' => ['nullable', 'string', 'max:50', \Illuminate\Validation\Rule::unique('warehouses')->ignore($warehouse->id)],
            'address' => 'nullable|string',
            'phone' => 'nullable|string|max:50',
            'is_default' => 'nullable|boolean',
            'is_active' => 'nullable|boolean',
            'notes' => 'nullable|string',
        ]);

        $warehouse->update($validated);

        return redirect()->route('warehouses.index')
            ->with('success', trans('messages.warehouse_updated'));
    }

    public function destroy(Warehouse $warehouse)
    {
        if (Setting::get('warehouse_mode', 'single') !== 'multiple') {
            return redirect()->route('settings.index')
                ->with('error', trans('messages.multiple_warehouses_not_enabled'));
        }

        // Cannot delete default warehouse
        if ($warehouse->is_default) {
            return back()->with('error', trans('messages.cannot_delete_default_warehouse'));
        }

        // Check if warehouse has stock
        $hasStock = $warehouse->products()->where('product_warehouse.quantity', '>', 0)->exists();
        if ($hasStock) {
            return back()->with('error', trans('messages.cannot_delete_warehouse_with_stock'));
        }

        DB::beginTransaction();
        try {
            // Delete all product-warehouse relationships
            $warehouse->products()->detach();
            
            // Delete warehouse
            $warehouse->delete();

            DB::commit();

            return redirect()->route('warehouses.index')
                ->with('success', trans('messages.warehouse_deleted'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.error_deleting_warehouse') . ': ' . $e->getMessage());
        }
    }
}
