<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use App\Services\LicenseService;
use App\Services\TrialService;
use Symfony\Component\HttpFoundation\Response;

class CheckLicense
{
    protected $licenseService;
    protected $trialService;

    public function __construct(LicenseService $licenseService, TrialService $trialService)
    {
        $this->licenseService = $licenseService;
        $this->trialService = $trialService;
    }

    public function handle(Request $request, Closure $next): Response
    {
        // الحصول على Route name
        $routeName = $request->route()?->getName();
        $path = $request->path();

        // السماح بصفحات الترخيص والفترة التجريبية
        if ($routeName && (str_starts_with($routeName, 'license.') || str_starts_with($routeName, 'trial.'))) {
            return $next($request);
        }

        // السماح بصفحات المصادقة
        if ($routeName && (in_array($routeName, ['login', 'logout']) || str_starts_with($routeName, 'password.'))) {
            return $next($request);
        }

        // السماح بصفحة اللغة
        if ($routeName === 'language.switch') {
            return $next($request);
        }

        // السماح بصفحة الرئيسية
        if ($path === '/' || $path === '') {
            return $next($request);
        }

        // التحقق من الترخيص أولاً
        $hasLicense = $this->licenseService->verifyLocal();
        if ($hasLicense) {
            return $next($request);
        }

        // التحقق من الفترة التجريبية
        try {
            $isExpired = $this->trialService->isExpired();
            $trialInfo = $this->trialService->getTrialInfo();
            
            // Logging للتشخيص (يمكن حذفه لاحقاً)
            \Log::info('Trial Check', [
                'route' => $routeName,
                'path' => $path,
                'is_expired' => $isExpired,
                'trial_info' => $trialInfo,
                'has_license' => $this->licenseService->verifyLocal()
            ]);
            
        } catch (\Exception $e) {
            // في حالة وجود خطأ، السماح بالمرور (للاستمرار في العمل)
            \Log::error('Trial check error: ' . $e->getMessage());
            return $next($request);
        }
        
        if (!$isExpired) {
            // عرض تحذير قبل 3 أيام من الانتهاء
            if ($this->trialService->isExpiringSoon()) {
                $daysLeft = $this->trialService->getDaysRemaining();
                if ($request->session()->get('trial_warning_shown') !== true) {
                    $request->session()->flash('trial_warning', [
                        'message' => "تنبيه: الفترة التجريبية ستنتهي خلال {$daysLeft} يوم/أيام. يرجى تفعيل الترخيص للاستمرار.",
                        'days_left' => $daysLeft
                    ]);
                    $request->session()->put('trial_warning_shown', true);
                }
            }
            
            return $next($request);
        }

        // إذا انتهت الفترة التجريبية وليس هناك ترخيص
        if ($request->expectsJson()) {
            return response()->json([
                'message' => 'انتهت الفترة التجريبية. يرجى تفعيل الترخيص.',
                'expired' => true,
                'license_required' => true
            ], 403);
        }

        // توجيه إلى صفحة انتهاء الفترة
        return redirect()->route('trial.expired');
    }
}
