<?php

namespace App\Http\Middleware;

use App\Models\ActivityLog;
use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class LogActivity
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        // Skip logging for certain routes
        $skipRoutes = ['api', 'login', 'logout', 'password', 'sanctum'];
        $currentRoute = $request->route()->getName() ?? $request->path();
        
        foreach ($skipRoutes as $skip) {
            if (str_contains($currentRoute, $skip)) {
                return $next($request);
            }
        }

        // Only log authenticated users
        if (!auth()->check()) {
            return $next($request);
        }

        $response = $next($request);

        // Log the activity after the request is processed
        try {
            $this->logActivity($request, $response);
        } catch (\Exception $e) {
            // Don't break the request if logging fails
            \Log::error('Activity log error: ' . $e->getMessage());
        }

        return $response;
    }

    private function logActivity(Request $request, Response $response): void
    {
        // Determine action based on HTTP method and route
        $action = $this->determineAction($request);
        
        // Skip if action is not loggable
        if (!$action) {
            return;
        }

        // Get model information from route parameters
        $modelInfo = $this->extractModelInfo($request);

        ActivityLog::create([
            'user_id' => auth()->id(),
            'action' => $action,
            'model_type' => $modelInfo['model_type'],
            'model_id' => $modelInfo['model_id'],
            'description' => $this->generateDescription($request, $action, $modelInfo),
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
            'route' => $request->route()->getName(),
            'method' => $request->method(),
        ]);
    }

    private function determineAction(Request $request): ?string
    {
        $method = $request->method();
        $routeName = $request->route()->getName() ?? '';

        // Map HTTP methods and route patterns to actions
        if ($method === 'POST' && str_contains($routeName, 'store')) {
            return 'created';
        } elseif ($method === 'PUT' || $method === 'PATCH') {
            if (str_contains($routeName, 'update')) {
                return 'updated';
            }
        } elseif ($method === 'DELETE') {
            return 'deleted';
        } elseif ($method === 'GET') {
            // Only log views for important pages
            $importantViews = ['show', 'edit', 'index'];
            foreach ($importantViews as $view) {
                if (str_contains($routeName, $view)) {
                    return 'viewed';
                }
            }
        }

        return null;
    }

    private function extractModelInfo(Request $request): array
    {
        $route = $request->route();
        $parameters = $route->parameters();

        // Try to find model from route parameters
        foreach ($parameters as $key => $value) {
            if (is_object($value)) {
                $class = get_class($value);
                if (str_starts_with($class, 'App\\Models\\')) {
                    return [
                        'model_type' => $class,
                        'model_id' => $value->id ?? null,
                    ];
                }
            }
        }

        // Try to infer from route name
        $routeName = $route->getName() ?? '';
        $modelMap = [
            'products' => 'App\\Models\\Product',
            'invoices' => 'App\\Models\\Invoice',
            'customers' => 'App\\Models\\Customer',
            'suppliers' => 'App\\Models\\Supplier',
            'users' => 'App\\Models\\User',
            'categories' => 'App\\Models\\Category',
        ];

        foreach ($modelMap as $key => $modelClass) {
            if (str_contains($routeName, $key)) {
                $id = $parameters[$key] ?? $parameters['id'] ?? null;
                return [
                    'model_type' => $modelClass,
                    'model_id' => $id,
                ];
            }
        }

        return [
            'model_type' => null,
            'model_id' => null,
        ];
    }

    private function generateDescription(Request $request, string $action, array $modelInfo): string
    {
        $routeName = $request->route()->getName() ?? '';
        $modelName = class_basename($modelInfo['model_type'] ?? '');
        
        if ($modelName) {
            return ucfirst($action) . ' ' . $modelName;
        }

        // Fallback to route name
        return ucfirst($action) . ' ' . str_replace('.', ' ', $routeName);
    }
}
