<?php

namespace App\Imports;

use App\Models\PurchaseInvoice;
use App\Models\Supplier;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\SkipsOnFailure;
use Maatwebsite\Excel\Concerns\SkipsFailures;

class PurchaseInvoicesImport implements ToCollection, WithHeadingRow, SkipsOnFailure
{
    use SkipsFailures;

    protected $imported = 0;
    protected $failed = 0;
    protected $errors = [];

    public function collection(Collection $rows)
    {
        $purchaseInvoicesToInsert = [];
        $rowNumber = 1;
        $existingNumbers = PurchaseInvoice::pluck('invoice_number')->toArray();
        $usedNumbers = [];

        if ($rows->isEmpty()) {
            return;
        }

        foreach ($rows as $row) {
            $rowNumber++;
            try {
                if (!($row instanceof Collection) && !is_array($row)) {
                    continue;
                }
                if (is_array($row)) {
                    $row = collect($row);
                }

                $invoiceNumber = $this->getColumnValue($row, ['invoice_number', 'Invoice Number', 'INVOICE_NUMBER']);
                if (empty($invoiceNumber)) {
                    throw new \Exception(trans('messages.invoice_number_required'));
                }

                if (in_array($invoiceNumber, $existingNumbers) || in_array($invoiceNumber, $usedNumbers)) {
                    throw new \Exception(trans('messages.duplicate_invoice_number'));
                }
                $usedNumbers[] = $invoiceNumber;

                $supplierId = $this->getColumnValue($row, ['supplier_id', 'Supplier ID', 'SUPPLIER_ID']);
                if (empty($supplierId)) {
                    throw new \Exception(trans('messages.supplier_required'));
                }

                $supplier = Supplier::find($supplierId);
                if (!$supplier) {
                    throw new \Exception(trans('messages.supplier_not_found'));
                }

                $purchaseDate = $this->getColumnValue($row, ['purchase_date', 'Purchase Date', 'PURCHASE_DATE']);
                if (empty($purchaseDate)) {
                    $purchaseDate = now()->format('Y-m-d');
                } else {
                    try {
                        $purchaseDate = \Carbon\Carbon::parse($purchaseDate)->format('Y-m-d');
                    } catch (\Exception $e) {
                        $purchaseDate = now()->format('Y-m-d');
                    }
                }

                $status = $this->getColumnValue($row, ['status', 'Status', 'STATUS']) ?? 'draft';
                $validStatuses = ['draft', 'confirmed', 'paid', 'partially_paid', 'returned'];
                if (!in_array($status, $validStatuses)) {
                    $status = 'draft';
                }

                $subtotal = $this->parseNumeric($this->getColumnValue($row, ['subtotal', 'Subtotal', 'SUBTOTAL'])) ?? 0;
                $tax = $this->parseNumeric($this->getColumnValue($row, ['tax', 'Tax', 'TAX'])) ?? 0;
                $discount = $this->parseNumeric($this->getColumnValue($row, ['discount', 'Discount', 'DISCOUNT'])) ?? 0;
                $totalAmount = $this->parseNumeric($this->getColumnValue($row, ['total_amount', 'Total Amount', 'TOTAL_AMOUNT']));
                $amountPaid = $this->parseNumeric($this->getColumnValue($row, ['amount_paid', 'Amount Paid', 'AMOUNT_PAID'])) ?? 0;

                // Calculate total if not provided
                if ($totalAmount === null) {
                    $totalAmount = $subtotal + $tax - $discount;
                }

                // Calculate amount remaining
                $amountRemaining = $totalAmount - $amountPaid;

                $purchaseInvoicesToInsert[] = [
                    'invoice_number' => trim($invoiceNumber),
                    'supplier_id' => (int)$supplierId,
                    'purchase_date' => $purchaseDate,
                    'status' => $status,
                    'subtotal' => $subtotal,
                    'tax' => $tax,
                    'discount' => $discount,
                    'total_amount' => $totalAmount,
                    'amount_paid' => $amountPaid,
                    'amount_remaining' => $amountRemaining,
                    'notes' => $this->getColumnValue($row, ['notes', 'Notes', 'NOTES']) ? trim($this->getColumnValue($row, ['notes', 'Notes', 'NOTES'])) : null,
                    'user_id' => auth()->id(),
                    'created_at' => now(),
                    'updated_at' => now(),
                ];
                $this->imported++;
            } catch (\Exception $e) {
                $this->failed++;
                $this->errors[] = [
                    'row' => $rowNumber,
                    'name' => $invoiceNumber ?? 'N/A',
                    'error' => $e->getMessage(),
                ];
            }
        }

        if (!empty($purchaseInvoicesToInsert)) {
            try {
                DB::beginTransaction();
                $chunks = array_chunk($purchaseInvoicesToInsert, 500);
                foreach ($chunks as $chunk) {
                    PurchaseInvoice::insert($chunk);
                }
                DB::commit();
            } catch (\Exception $e) {
                DB::rollBack();
                $this->failed += count($purchaseInvoicesToInsert);
                $this->imported -= count($purchaseInvoicesToInsert);
                $this->errors[] = [
                    'row' => 'Bulk Insert',
                    'name' => 'Multiple Purchase Invoices',
                    'error' => 'Database error during bulk insert: ' . $e->getMessage(),
                ];
            }
        }
    }

    private function getColumnValue($row, array $possibleNames)
    {
        foreach ($possibleNames as $name) {
            $lowerName = strtolower(trim($name));
            foreach ($row->keys() as $key) {
                $lowerKey = strtolower(trim($key));
                if ($lowerKey === $lowerName) {
                    $value = $row->get($key);
                    if ($value !== null && $value !== '' && $value !== []) {
                        return is_string($value) ? trim($value) : $value;
                    }
                    return null;
                }
            }
        }
        return null;
    }

    private function parseNumeric($value)
    {
        if ($value === null || $value === '') {
            return null;
        }
        if (is_numeric($value)) {
            return (float)$value;
        }
        $cleaned = preg_replace('/[^0-9.]/', '', (string)$value);
        return !empty($cleaned) ? (float)$cleaned : null;
    }

    public function getImported() { return $this->imported; }
    public function getFailed() { return $this->failed; }
    public function getErrors() { return $this->errors; }
}

