<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;

class Invoice extends Model
{
    use HasFactory;

    protected static function newFactory()
    {
        return \Database\Factories\InvoiceFactory::new();
    }

    protected $fillable = [
        'invoice_number',
        'quotation_number',
        'customer_id',
        'customer_name',
        'status',
        'subtotal',
        'discount',
        'tax',
        'total',
        'paid_amount',
        'due_amount',
        'notes',
        'user_id',
    ];

    protected function casts(): array
    {
        return [
            'subtotal' => 'decimal:2',
            'discount' => 'decimal:2',
            'tax' => 'decimal:2',
            'total' => 'decimal:2',
            'paid_amount' => 'decimal:2',
            'due_amount' => 'decimal:2',
        ];
    }

    public function customer()
    {
        return $this->belongsTo(Customer::class);
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function items()
    {
        return $this->hasMany(InvoiceItem::class);
    }

    public function payments()
    {
        return $this->hasMany(Payment::class);
    }

    public function scopeDraft(Builder $query)
    {
        return $query->where('status', 'draft');
    }

    public function scopeFinal(Builder $query)
    {
        return $query->where('status', 'final');
    }

    public function scopePaid(Builder $query)
    {
        return $query->where('status', 'paid');
    }

    public function scopePending(Builder $query)
    {
        return $query->whereIn('status', ['final', 'partially_paid']);
    }

    public function isPaid(): bool
    {
        return $this->status === 'paid' || $this->due_amount <= 0;
    }

    public function isDraft(): bool
    {
        return $this->status === 'draft';
    }

    public function isReturned(): bool
    {
        return $this->status === 'returned';
    }

    public function updateTotals(): void
    {
        $subtotal = $this->items->sum('total');
        $discount = $this->discount;
        $tax = $this->items->sum(function ($item) {
            return ($item->subtotal - $item->discount) * ($item->tax_rate / 100);
        });

        $this->subtotal = $subtotal;
        $this->tax = $tax;
        $this->total = $subtotal + $tax - $discount;
        $this->due_amount = $this->total - $this->paid_amount;

        // Update status based on payment amount
        if ($this->due_amount <= 0 && $this->paid_amount > 0) {
            $this->status = 'paid';
        } elseif ($this->paid_amount > 0 && $this->paid_amount < $this->total) {
            $this->status = 'partially_paid';
        } elseif ($this->paid_amount == 0 && !$this->isDraft() && !$this->isReturned()) {
            // If all payments are deleted and invoice is not draft or returned, set status to final
            $this->status = 'final';
        }

        $this->save();
    }
}

