<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class PurchaseInvoice extends Model
{
    use HasFactory;

    protected $fillable = [
        'supplier_id',
        'invoice_number',
        'purchase_date',
        'status',
        'subtotal',
        'tax',
        'discount',
        'total_amount',
        'amount_paid',
        'amount_remaining',
        'notes',
        'supplier_invoice_file',
        'user_id',
        'warehouse_id',
    ];

    protected $casts = [
        'purchase_date' => 'date',
        'subtotal' => 'decimal:2',
        'tax' => 'decimal:2',
        'discount' => 'decimal:2',
        'total_amount' => 'decimal:2',
        'amount_paid' => 'decimal:2',
        'amount_remaining' => 'decimal:2',
    ];

    public function supplier(): BelongsTo
    {
        return $this->belongsTo(Supplier::class);
    }

    public function items(): HasMany
    {
        return $this->hasMany(PurchaseInvoiceItem::class);
    }

    public function payments(): HasMany
    {
        return $this->hasMany(SupplierPayment::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function warehouse(): BelongsTo
    {
        return $this->belongsTo(Warehouse::class);
    }

    public function isDraft(): bool
    {
        return $this->status === 'draft';
    }

    public function isConfirmed(): bool
    {
        return $this->status === 'confirmed';
    }

    public function isPaid(): bool
    {
        return $this->status === 'paid';
    }

    public function isReturned(): bool
    {
        return $this->status === 'returned';
    }

    public function updateTotals(): void
    {
        $this->subtotal = $this->items->sum('total');
        $this->total_amount = $this->subtotal - $this->discount + $this->tax;
        $this->amount_paid = $this->payments->sum('amount_paid');
        $this->amount_remaining = $this->total_amount - $this->amount_paid;
        
        // Update status based on payments (same logic as sales invoices)
        if ($this->amount_remaining <= 0 && $this->amount_paid > 0) {
            // If amount remaining is 0, status is paid
            $this->status = 'paid';
        } elseif ($this->amount_paid > 0 && $this->amount_paid < $this->total_amount) {
            // If total amount != amount remaining (i.e., some payment made), status is partially_paid
            $this->status = 'partially_paid';
        } elseif ($this->amount_paid == 0 && !$this->isDraft() && !$this->isReturned()) {
            // If total amount == amount remaining (i.e., no payment made), status is confirmed
            $this->status = 'confirmed';
        }
        
        $this->save();
    }

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($invoice) {
            if (empty($invoice->invoice_number)) {
                // Get purchase invoice number format from settings
                $format = Setting::get('purchase_invoice_number_format', 'PINV-{YEAR}-{NUMBER}');
                
                // Replace {YEAR} with current year
                $year = date('Y');
                $formatWithYear = str_replace('{YEAR}', $year, $format);
                
                // Extract prefix from format (everything before {NUMBER})
                $prefix = str_replace('{NUMBER}', '', $formatWithYear);
                
                // Find all purchase invoices that match the prefix pattern for current year
                $matchingInvoices = static::where('invoice_number', 'like', $prefix . '%')
                    ->get();
                
                // Extract numbers from matching invoices
                $numbers = [];
                foreach ($matchingInvoices as $inv) {
                    // Extract numeric part from the end
                    $invoiceNum = $inv->invoice_number;
                    // Remove prefix to get the number part
                    $numberPart = str_replace($prefix, '', $invoiceNum);
                    // Extract only numeric characters
                    preg_match('/\d+/', $numberPart, $matches);
                    if (!empty($matches)) {
                        $numbers[] = (int)$matches[0];
                    }
                }
                
                // Get the next number
                $number = empty($numbers) ? 1 : max($numbers) + 1;
                
                // Replace {NUMBER} with the generated number (padded to 6 digits)
                $invoiceNumber = str_replace('{NUMBER}', str_pad($number, 6, '0', STR_PAD_LEFT), $formatWithYear);
                
                // Ensure uniqueness
                $counter = 0;
                while (static::where('invoice_number', $invoiceNumber)->exists() && $counter < 1000) {
                    $number++;
                    $invoiceNumber = str_replace('{NUMBER}', str_pad($number, 6, '0', STR_PAD_LEFT), $formatWithYear);
                    $counter++;
                }
                
                $invoice->invoice_number = $invoiceNumber;
            }
        });
    }
}

