<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;

class Setting extends Model
{
    use HasFactory;

    protected $fillable = [
        'key',
        'value',
    ];

    public static function get($key, $default = null)
    {
        return Cache::remember("setting.{$key}", 3600, function () use ($key, $default) {
            $setting = static::where('key', $key)->first();
            return $setting ? $setting->value : $default;
        });
    }

    public static function getItemsPerPage($default = 20)
    {
        return (int)Cache::remember("setting.items_per_page", 3600, function () use ($default) {
            $setting = static::where('key', 'items_per_page')->first();
            return $setting ? (int)$setting->value : $default;
        });
    }

    public static function set($key, $value): void
    {
        static::updateOrCreate(
            ['key' => $key],
            ['value' => (string)$value]
        );
        Cache::forget("setting.{$key}");
        // Also clear items_per_page cache if that's what we're updating
        if ($key === 'items_per_page') {
            Cache::forget('setting.items_per_page');
        }
    }

    public static function forget($key): void
    {
        static::where('key', $key)->delete();
        Cache::forget("setting.{$key}");
    }

    public static function clearCache(): void
    {
        Cache::flush();
    }
}

