<?php

namespace App\Services;

use App\Models\License;
use App\Models\Setting;
use App\Services\MachineIdService;
use App\Services\LicenseDatabaseService;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class LicenseService
{
    protected $machineIdService;
    protected $licenseDb;

    public function __construct(MachineIdService $machineIdService, LicenseDatabaseService $licenseDb)
    {
        $this->machineIdService = $machineIdService;
        $this->licenseDb = $licenseDb;
    }

    /**
     * التحقق من الترخيص محلياً
     */
    public function verifyLocal(): bool
    {
        // البحث في قاعدة البيانات المنفصلة
        $currentMachineId = $this->machineIdService->getMachineId();
        $licenseData = $this->licenseDb->findByMachineId($currentMachineId);
        
        if (!$licenseData) {
            return false;
        }

        // التحقق من الحالة
        if ($licenseData['status'] !== 'active') {
            return false;
        }

        // التحقق من تاريخ الانتهاء
        if (isset($licenseData['expires_at']) && $licenseData['expires_at']) {
            $expiresAt = Carbon::parse($licenseData['expires_at']);
            if (Carbon::now()->greaterThan($expiresAt)) {
                return false;
            }
        }

        return true;
    }

    /**
     * مفتاح التشفير السري (يجب تغييره في الإنتاج)
     * يجب حفظه في .env كـ LICENSE_SECRET_KEY
     */
    protected function getSecretKey(): string
    {
        return config('app.license_secret_key', env('LICENSE_SECRET_KEY', 'CHANGE_THIS_SECRET_KEY_IN_PRODUCTION'));
    }

    /**
     * إنشاء Hash لمفتاح الترخيص
     */
    protected function generateLicenseHash(string $licenseKey, string $serialNumber): string
    {
        $secret = $this->getSecretKey();
        $data = $licenseKey . $serialNumber . $secret;
        return hash('sha256', $data);
    }

    /**
     * التحقق من صحة توقيع مفتاح الترخيص
     */
    protected function verifyLicenseSignature(string $licenseKey, string $serialNumber, string $signature): bool
    {
        $expectedHash = $this->generateLicenseHash($licenseKey, $serialNumber);
        return hash_equals($expectedHash, $signature);
    }

    /**
     * التحقق من صحة تنسيق مفتاح الترخيص
     */
    protected function validateLicenseFormat(string $licenseKey, string $serialNumber): bool
    {
        // تنسيق License Key: يجب أن يكون على شكل: XXXX-XXXX-XXXX-XXXX
        $licensePattern = '/^[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}$/';
        
        // تنسيق Serial Number: يجب أن يبدأ بـ SN- ثم 12 حرف/رقم
        $serialPattern = '/^SN-[A-Z0-9]{12}$/';
        
        $licenseValid = preg_match($licensePattern, strtoupper($licenseKey));
        $serialValid = preg_match($serialPattern, strtoupper($serialNumber));
        
        return $licenseValid && $serialValid;
    }

    /**
     * التحقق من الترخيص (نظام آمن)
     */
    public function verifyLicense(string $licenseKey, string $serialNumber, ?string $signature = null): array
    {
        // 1. التحقق من التنسيق أولاً
        if (!$this->validateLicenseFormat($licenseKey, $serialNumber)) {
            return [
                'valid' => false,
                'message' => 'تنسيق مفتاح الترخيص أو الرقم التسلسلي غير صحيح',
            ];
        }

        // 2. للاختبار فقط: قبول مفاتيح TEST/DEMO (فقط في بيئة التطوير)
        if (app()->environment('local', 'testing')) {
            if (strpos(strtoupper($licenseKey), 'TEST') !== false || 
                strpos(strtoupper($licenseKey), 'DEMO') !== false) {
                return [
                    'valid' => true,
                    'data' => [
                        'status' => 'active',
                        'expires_at' => null,
                        'max_users' => 10,
                    ],
                ];
            }
        }

        // 3. التحقق من قاعدة البيانات المنفصلة - يجب أن يكون الترخيص موجوداً مسبقاً
        $licenseData = $this->licenseDb->find(strtoupper($licenseKey), strtoupper($serialNumber));

        if (!$licenseData) {
            // الترخيص غير موجود في قاعدة البيانات المنفصلة
            // هذا يعني أنه لم يتم إصداره من قبل النظام
            return [
                'valid' => false,
                'message' => 'مفتاح الترخيص غير موجود. يرجى التواصل مع الدعم للحصول على ترخيص صحيح.',
            ];
        }

        // 4. التحقق من صحة الترخيص
        if ($licenseData['status'] !== 'active' && $licenseData['status'] !== 'pending') {
            return [
                'valid' => false,
                'message' => 'الترخيص غير مفعّل أو معلّق',
            ];
        }

        // التحقق من تاريخ الانتهاء
        if (isset($licenseData['expires_at']) && $licenseData['expires_at']) {
            $expiresAt = Carbon::parse($licenseData['expires_at']);
            if (Carbon::now()->greaterThan($expiresAt)) {
                return [
                    'valid' => false,
                    'message' => 'الترخيص منتهي الصلاحية',
                ];
            }
        }

        // 5. التحقق من التوقيع إذا كان موجوداً
        if ($signature && !$this->verifyLicenseSignature($licenseKey, $serialNumber, $signature)) {
            return [
                'valid' => false,
                'message' => 'توقيع الترخيص غير صحيح',
            ];
        }

        // 6. التحقق من أن الترخيص لم يتم استخدامه على نطاق آخر
        if (isset($licenseData['domain']) && $licenseData['domain'] && $licenseData['domain'] !== request()->getHost()) {
            return [
                'valid' => false,
                'message' => 'هذا الترخيص مفعّل على نطاق آخر. يرجى التواصل مع الدعم.',
            ];
        }

        // 7. التحقق من Machine ID - الترخيص مربوط بجهاز واحد فقط
        $currentMachineId = $this->machineIdService->getMachineId();
        if (isset($licenseData['machine_id']) && $licenseData['machine_id'] && $licenseData['machine_id'] !== $currentMachineId) {
            return [
                'valid' => false,
                'message' => 'هذا الترخيص مفعّل على جهاز آخر. كل ترخيص يعمل على جهاز واحد فقط. يرجى التواصل مع الدعم لنقل الترخيص.',
            ];
        }

        return [
            'valid' => true,
            'data' => [
                'status' => $licenseData['status'],
                'expires_at' => isset($licenseData['expires_at']) ? $licenseData['expires_at'] : null,
                'max_users' => $licenseData['max_users'] ?? 10,
            ],
        ];
    }

    /**
     * تفعيل الترخيص
     */
    public function activate(string $licenseKey, string $serialNumber, array $customerData = []): array
    {
        // التحقق من الترخيص أولاً
        $verification = $this->verifyLicense($licenseKey, $serialNumber);

        if (!$verification['valid']) {
            return [
                'success' => false,
                'message' => $verification['message'] ?? 'مفتاح الترخيص غير صحيح',
            ];
        }

        $verificationData = $verification['data'];

        // البحث عن الترخيص في قاعدة البيانات المنفصلة (يجب أن يكون موجوداً مسبقاً)
        $existingLicense = $this->licenseDb->find(strtoupper($licenseKey), strtoupper($serialNumber));

        if (!$existingLicense) {
            // هذا لا يجب أن يحدث إذا كان verifyLicense يعمل بشكل صحيح
            return [
                'success' => false,
                'message' => 'خطأ في التحقق من الترخيص. يرجى التواصل مع الدعم.',
            ];
        }

        // الحصول على Machine ID للجهاز الحالي
        $machineId = $this->machineIdService->getMachineId();
        $machineName = $this->machineIdService->getMachineName();
        $ipAddress = request()->ip();

        // تحديث معلومات التفعيل في قاعدة البيانات المنفصلة
        $updateData = [
            'domain' => request()->getHost(),
            'machine_id' => $machineId,
            'machine_name' => $machineName,
            'ip_address' => $ipAddress,
            'email' => $customerData['email'] ?? $existingLicense['email'],
            'customer_name' => $customerData['name'] ?? $existingLicense['customer_name'],
            'status' => 'active',
            'activated_at' => Carbon::now()->format('Y-m-d H:i:s'),
            'expires_at' => isset($verificationData['expires_at']) ? $verificationData['expires_at'] : $existingLicense['expires_at'],
            'max_users' => $verificationData['max_users'] ?? $existingLicense['max_users'] ?? 10,
        ];

        $this->licenseDb->update(strtoupper($licenseKey), $updateData);

        // حفظ في Settings (فقط معلومات عامة، بدون البيانات الحساسة)
        Setting::set('license_key', substr(strtoupper($licenseKey), 0, 8) . '****');
        Setting::set('license_status', 'active');
        Setting::set('license_activated_at', Carbon::now()->format('Y-m-d'));

        // الحصول على بيانات الترخيص المحدثة
        $updatedLicense = $this->licenseDb->find(strtoupper($licenseKey), strtoupper($serialNumber));

        return [
            'success' => true,
            'message' => 'تم تفعيل الترخيص بنجاح',
            'license' => $updatedLicense,
        ];
    }

    /**
     * الحصول على معلومات الترخيص الحالي
     */
    public function getLicenseInfo(): ?array
    {
        $currentMachineId = $this->machineIdService->getMachineId();
        $licenseData = $this->licenseDb->findByMachineId($currentMachineId);

        if (!$licenseData) {
            return null;
        }

        $isValid = $licenseData['status'] === 'active';
        
        // التحقق من تاريخ الانتهاء
        if ($isValid && isset($licenseData['expires_at']) && $licenseData['expires_at']) {
            $expiresAt = Carbon::parse($licenseData['expires_at']);
            if (Carbon::now()->greaterThan($expiresAt)) {
                $isValid = false;
            }
        }

        // حساب الأيام المتبقية
        $daysRemaining = null;
        if (isset($licenseData['expires_at']) && $licenseData['expires_at']) {
            $expiresAt = Carbon::parse($licenseData['expires_at']);
            $daysRemaining = Carbon::now()->diffInDays($expiresAt, false);
            if ($daysRemaining < 0) {
                $daysRemaining = 0;
            }
        }

        return [
            'license_key' => substr($licenseData['license_key'], 0, 8) . '****', // إخفاء جزء من المفتاح
            'serial_number' => $licenseData['serial_number'],
            'customer_name' => $licenseData['customer_name'] ?? null,
            'email' => $licenseData['email'] ?? null,
            'status' => $licenseData['status'],
            'activated_at' => isset($licenseData['activated_at']) ? Carbon::parse($licenseData['activated_at'])->format('Y-m-d') : null,
            'expires_at' => isset($licenseData['expires_at']) ? Carbon::parse($licenseData['expires_at'])->format('Y-m-d') : null,
            'days_remaining' => $daysRemaining,
            'max_users' => $licenseData['max_users'] ?? 10,
            'is_valid' => $isValid,
            'machine_id' => isset($licenseData['machine_id']) && $licenseData['machine_id'] ? substr($licenseData['machine_id'], 0, 8) . '****' : null,
            'machine_name' => $licenseData['machine_name'] ?? null,
            'is_machine_match' => true, // إذا وصلنا هنا، يعني Machine ID متطابق
        ];
    }

    /**
     * إلغاء تفعيل الترخيص
     */
    public function deactivate(): void
    {
        $currentMachineId = $this->machineIdService->getMachineId();
        $licenseData = $this->licenseDb->findByMachineId($currentMachineId);
        
        if ($licenseData) {
            $this->licenseDb->update($licenseData['license_key'], ['status' => 'suspended']);
        }
        
        Setting::set('license_status', 'inactive');
    }

    /**
     * نقل الترخيص من جهاز لآخر (للمطور فقط)
     */
    public function transferLicense(string $licenseKey, bool $force = false): array
    {
        $licenseData = $this->licenseDb->find(strtoupper($licenseKey));

        if (!$licenseData) {
            return [
                'success' => false,
                'message' => 'الترخيص غير موجود',
            ];
        }

        if (!$force && isset($licenseData['machine_id']) && $licenseData['machine_id']) {
            return [
                'success' => false,
                'message' => 'الترخيص مفعّل على جهاز آخر. استخدم --force للنقل القسري.',
            ];
        }

        $machineId = $this->machineIdService->getMachineId();
        $machineName = $this->machineIdService->getMachineName();

        $this->licenseDb->update(strtoupper($licenseKey), [
            'machine_id' => $machineId,
            'machine_name' => $machineName,
            'ip_address' => request()->ip(),
        ]);

        $updatedLicense = $this->licenseDb->find(strtoupper($licenseKey));

        return [
            'success' => true,
            'message' => 'تم نقل الترخيص بنجاح',
            'license' => $updatedLicense,
        ];
    }
}

