<?php

namespace App\Services;

use App\Models\Notification;
use App\Models\Product;
use App\Models\ProductBatch;
use App\Models\Invoice;
use App\Models\Cheque;
use App\Models\PromissoryNote;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class NotificationService
{
    /**
     * Generate all notifications
     */
    public function generateAllNotifications(): void
    {
        $this->checkLowStockProducts();
        $this->checkExpiringBatches();
        $this->checkDueInvoices();
        $this->checkDueCheques();
        $this->checkDuePromissoryNotes();
    }

    /**
     * Check for low stock products
     */
    public function checkLowStockProducts(): void
    {
        $lowStockProducts = Product::lowStock()->get();
        
        foreach ($lowStockProducts as $product) {
            // Check if notification already exists and is not read
            $exists = Notification::where('type', 'low_stock')
                ->where('model_type', Product::class)
                ->where('model_id', $product->id)
                ->where('is_read', false)
                ->exists();

            if (!$exists) {
                Notification::create([
                    'type' => 'low_stock',
                    'title' => trans('messages.low_stock_notification_title', ['product' => $product->name]),
                    'message' => trans('messages.low_stock_notification_message', [
                        'product' => $product->name,
                        'stock' => $product->stock_quantity,
                        'reorder' => $product->reorder_level
                    ]),
                    'severity' => 'warning',
                    'icon' => 'bi-exclamation-triangle',
                    'model_type' => Product::class,
                    'model_id' => $product->id,
                    'data' => [
                        'product_id' => $product->id,
                        'stock_quantity' => $product->stock_quantity,
                        'reorder_level' => $product->reorder_level,
                    ],
                ]);
            }
        }

        // Remove notifications for products that are no longer low stock
        $productIds = $lowStockProducts->pluck('id')->toArray();
        Notification::where('type', 'low_stock')
            ->where('model_type', Product::class)
            ->whereNotIn('model_id', $productIds)
            ->where('is_read', false)
            ->delete();
    }

    /**
     * Check for expiring batches
     */
    public function checkExpiringBatches(int $daysBeforeExpiry = 30): void
    {
        $expiringDate = Carbon::now()->addDays($daysBeforeExpiry);
        
        $expiringBatches = ProductBatch::whereNotNull('expiry_date')
            ->where('expiry_date', '<=', $expiringDate)
            ->where('expiry_date', '>=', Carbon::now())
            ->where('current_quantity', '>', 0)
            ->with('product')
            ->get();

        foreach ($expiringBatches as $batch) {
            $daysUntilExpiry = Carbon::now()->diffInDays($batch->expiry_date, false);
            
            $exists = Notification::where('type', 'batch_expiring')
                ->where('model_type', ProductBatch::class)
                ->where('model_id', $batch->id)
                ->where('is_read', false)
                ->exists();

            if (!$exists) {
                $severity = $daysUntilExpiry <= 7 ? 'danger' : ($daysUntilExpiry <= 15 ? 'warning' : 'info');
                
                Notification::create([
                    'type' => 'batch_expiring',
                    'title' => trans('messages.batch_expiring_notification_title', [
                        'product' => $batch->product->name,
                        'batch' => $batch->batch_number
                    ]),
                    'message' => trans('messages.batch_expiring_notification_message', [
                        'product' => $batch->product->name,
                        'batch' => $batch->batch_number,
                        'days' => $daysUntilExpiry,
                        'expiry_date' => $batch->expiry_date->format('Y-m-d')
                    ]),
                    'severity' => $severity,
                    'icon' => 'bi-calendar-x',
                    'model_type' => ProductBatch::class,
                    'model_id' => $batch->id,
                    'data' => [
                        'product_id' => $batch->product_id,
                        'batch_id' => $batch->id,
                        'expiry_date' => $batch->expiry_date->format('Y-m-d'),
                        'days_until_expiry' => $daysUntilExpiry,
                    ],
                ]);
            }
        }

        // Remove notifications for batches that are no longer expiring
        $batchIds = $expiringBatches->pluck('id')->toArray();
        Notification::where('type', 'batch_expiring')
            ->where('model_type', ProductBatch::class)
            ->whereNotIn('model_id', $batchIds)
            ->where('is_read', false)
            ->delete();
    }

    /**
     * Check for due invoices
     */
    public function checkDueInvoices(): void
    {
        $dueInvoices = Invoice::where('status', '!=', 'draft')
            ->where('status', '!=', 'returned')
            ->where('due_amount', '>', 0)
            ->with('customer')
            ->get();

        foreach ($dueInvoices as $invoice) {
            $exists = Notification::where('type', 'invoice_due')
                ->where('model_type', Invoice::class)
                ->where('model_id', $invoice->id)
                ->where('is_read', false)
                ->exists();

            if (!$exists) {
                Notification::create([
                    'type' => 'invoice_due',
                    'title' => trans('messages.invoice_due_notification_title', [
                        'invoice' => $invoice->invoice_number
                    ]),
                    'message' => trans('messages.invoice_due_notification_message', [
                        'invoice' => $invoice->invoice_number,
                        'customer' => $invoice->customer?->name ?? $invoice->customer_name,
                        'amount' => format_currency($invoice->due_amount)
                    ]),
                    'severity' => 'warning',
                    'icon' => 'bi-receipt',
                    'model_type' => Invoice::class,
                    'model_id' => $invoice->id,
                    'data' => [
                        'invoice_id' => $invoice->id,
                        'invoice_number' => $invoice->invoice_number,
                        'due_amount' => $invoice->due_amount,
                    ],
                ]);
            }
        }

        // Remove notifications for invoices that are no longer due
        $invoiceIds = $dueInvoices->pluck('id')->toArray();
        Notification::where('type', 'invoice_due')
            ->where('model_type', Invoice::class)
            ->whereNotIn('model_id', $invoiceIds)
            ->where('is_read', false)
            ->delete();
    }

    /**
     * Check for due cheques
     */
    public function checkDueCheques(): void
    {
        $dueCheques = Cheque::where('status', 'pending')
            ->whereNotNull('due_date')
            ->where('due_date', '<=', Carbon::now()->addDays(7))
            ->with(['customer', 'supplier'])
            ->get();

        foreach ($dueCheques as $cheque) {
            $daysUntilDue = Carbon::now()->diffInDays($cheque->due_date, false);
            
            $exists = Notification::where('type', 'cheque_due')
                ->where('model_type', Cheque::class)
                ->where('model_id', $cheque->id)
                ->where('is_read', false)
                ->exists();

            if (!$exists) {
                $severity = $daysUntilDue < 0 ? 'danger' : ($daysUntilDue <= 3 ? 'warning' : 'info');
                $party = $cheque->customer ? $cheque->customer->name : ($cheque->supplier ? $cheque->supplier->name : 'N/A');
                
                Notification::create([
                    'type' => 'cheque_due',
                    'title' => trans('messages.cheque_due_notification_title', [
                        'cheque' => $cheque->cheque_number
                    ]),
                    'message' => trans('messages.cheque_due_notification_message', [
                        'cheque' => $cheque->cheque_number,
                        'party' => $party,
                        'amount' => format_currency($cheque->amount),
                        'due_date' => $cheque->due_date->format('Y-m-d'),
                        'days' => abs($daysUntilDue)
                    ]),
                    'severity' => $severity,
                    'icon' => 'bi-bank',
                    'model_type' => Cheque::class,
                    'model_id' => $cheque->id,
                    'data' => [
                        'cheque_id' => $cheque->id,
                        'cheque_number' => $cheque->cheque_number,
                        'amount' => $cheque->amount,
                        'due_date' => $cheque->due_date->format('Y-m-d'),
                        'days_until_due' => $daysUntilDue,
                    ],
                ]);
            }
        }

        // Remove notifications for cheques that are no longer due
        $chequeIds = $dueCheques->pluck('id')->toArray();
        Notification::where('type', 'cheque_due')
            ->where('model_type', Cheque::class)
            ->whereNotIn('model_id', $chequeIds)
            ->where('is_read', false)
            ->delete();
    }

    /**
     * Check for due promissory notes
     */
    public function checkDuePromissoryNotes(): void
    {
        $dueNotes = PromissoryNote::where('status', 'pending')
            ->whereNotNull('due_date')
            ->where('due_date', '<=', Carbon::now()->addDays(7))
            ->with(['customer', 'supplier'])
            ->get();

        foreach ($dueNotes as $note) {
            $daysUntilDue = Carbon::now()->diffInDays($note->due_date, false);
            
            $exists = Notification::where('type', 'promissory_note_due')
                ->where('model_type', PromissoryNote::class)
                ->where('model_id', $note->id)
                ->where('is_read', false)
                ->exists();

            if (!$exists) {
                $severity = $daysUntilDue < 0 ? 'danger' : ($daysUntilDue <= 3 ? 'warning' : 'info');
                $party = $note->customer ? $note->customer->name : ($note->supplier ? $note->supplier->name : 'N/A');
                
                Notification::create([
                    'type' => 'promissory_note_due',
                    'title' => trans('messages.promissory_note_due_notification_title', [
                        'note' => $note->note_number
                    ]),
                    'message' => trans('messages.promissory_note_due_notification_message', [
                        'note' => $note->note_number,
                        'party' => $party,
                        'amount' => format_currency($note->amount),
                        'due_date' => $note->due_date->format('Y-m-d'),
                        'days' => abs($daysUntilDue)
                    ]),
                    'severity' => $severity,
                    'icon' => 'bi-file-text',
                    'model_type' => PromissoryNote::class,
                    'model_id' => $note->id,
                    'data' => [
                        'note_id' => $note->id,
                        'note_number' => $note->note_number,
                        'amount' => $note->amount,
                        'due_date' => $note->due_date->format('Y-m-d'),
                        'days_until_due' => $daysUntilDue,
                    ],
                ]);
            }
        }

        // Remove notifications for notes that are no longer due
        $noteIds = $dueNotes->pluck('id')->toArray();
        Notification::where('type', 'promissory_note_due')
            ->where('model_type', PromissoryNote::class)
            ->whereNotIn('model_id', $noteIds)
            ->where('is_read', false)
            ->delete();
    }

    /**
     * Get unread notifications count
     */
    public function getUnreadCount(): int
    {
        return Notification::where('is_read', false)->count();
    }

    /**
     * Get recent notifications
     */
    public function getRecentNotifications(int $limit = 10)
    {
        return Notification::orderBy('created_at', 'desc')
            ->limit($limit)
            ->get();
    }
}

