<?php

namespace App\Services;

use App\Models\Product;
use App\Models\Warehouse;
use App\Models\StockMovement;
use App\Models\Setting;
use Illuminate\Support\Facades\DB;

class StockManager
{
    /**
     * Check if multiple warehouses mode is enabled
     */
    public static function isMultipleWarehouses(): bool
    {
        return Setting::get('warehouse_mode', 'single') === 'multiple';
    }

    /**
     * Get stock quantity for a product
     */
    public static function getStockQuantity(Product $product, $warehouseId = null): int
    {
        if (!self::isMultipleWarehouses()) {
            return $product->stock_quantity;
        }

        if ($warehouseId) {
            $pivot = $product->warehouses()->where('warehouses.id', $warehouseId)->first();
            return $pivot ? $pivot->pivot->quantity : 0;
        }

        // Return total from all warehouses
        return $product->warehouses()->sum('product_warehouse.quantity');
    }

    /**
     * Get default warehouse
     */
    public static function getDefaultWarehouse()
    {
        if (!self::isMultipleWarehouses()) {
            return null;
        }

        return Warehouse::getDefault();
    }

    /**
     * Update stock quantity
     */
    public static function updateStock(
        Product $product,
        int $quantity,
        string $type, // 'IN' or 'OUT'
        $warehouseId = null,
        $referenceType = null,
        $referenceId = null,
        $notes = null
    ): void {
        if (!self::isMultipleWarehouses()) {
            // Single warehouse mode - update product stock_quantity directly
            $quantityBefore = $product->stock_quantity;
            $product->stock_quantity += ($type === 'IN' ? $quantity : -$quantity);
            $product->stock_quantity = max(0, $product->stock_quantity);
            $product->save();

            // Create stock movement
            StockMovement::create([
                'product_id' => $product->id,
                'warehouse_id' => null,
                'type' => $type,
                'quantity' => $quantity,
                'quantity_before' => $quantityBefore,
                'quantity_after' => $product->stock_quantity,
                'reference_type' => $referenceType,
                'reference_id' => $referenceId,
                'notes' => $notes,
                'user_id' => auth()->id(),
            ]);
        } else {
            // Multiple warehouses mode
            if (!$warehouseId) {
                $warehouse = self::getDefaultWarehouse();
                if (!$warehouse) {
                    throw new \Exception('No default warehouse found');
                }
                $warehouseId = $warehouse->id;
            }

            $warehouse = Warehouse::findOrFail($warehouseId);

            // Get or create pivot record
            $pivot = $product->warehouses()->where('warehouses.id', $warehouseId)->first();
            
            if (!$pivot) {
                // Create pivot with initial values
                $product->warehouses()->attach($warehouseId, [
                    'quantity' => 0,
                    'reorder_level' => $product->reorder_level
                ]);
                $quantityBefore = 0;
            } else {
                $quantityBefore = $pivot->pivot->quantity;
            }

            // Calculate new quantity
            $newQuantity = $quantityBefore + ($type === 'IN' ? $quantity : -$quantity);
            $newQuantity = max(0, $newQuantity);

            // Update pivot
            $product->warehouses()->updateExistingPivot($warehouseId, [
                'quantity' => $newQuantity
            ]);

            // Create stock movement
            StockMovement::create([
                'product_id' => $product->id,
                'warehouse_id' => $warehouseId,
                'type' => $type,
                'quantity' => $quantity,
                'quantity_before' => $quantityBefore,
                'quantity_after' => $newQuantity,
                'reference_type' => $referenceType,
                'reference_id' => $referenceId,
                'notes' => $notes,
                'user_id' => auth()->id(),
            ]);
        }
    }

    /**
     * Check stock availability
     */
    public static function checkStockAvailability(
        Product $product,
        int $requiredQuantity,
        $warehouseId = null,
        bool $allowNegative = false
    ): bool {
        if ($allowNegative) {
            return true;
        }

        $availableQuantity = self::getStockQuantity($product, $warehouseId);
        return $availableQuantity >= $requiredQuantity;
    }

    /**
     * Get total stock across all warehouses
     */
    public static function getTotalStock(Product $product): int
    {
        if (!self::isMultipleWarehouses()) {
            return $product->stock_quantity;
        }

        return $product->warehouses()->sum('product_warehouse.quantity');
    }

    /**
     * Transfer stock between warehouses
     */
    public static function transferStock(
        Product $product,
        int $fromWarehouseId,
        int $toWarehouseId,
        int $quantity,
        $notes = null
    ): void {
        if (!self::isMultipleWarehouses()) {
            throw new \Exception('Transfer is only available in multiple warehouses mode');
        }

        // Check availability in source warehouse
        $availableQuantity = self::getStockQuantity($product, $fromWarehouseId);
        if ($availableQuantity < $quantity) {
            throw new \Exception("Insufficient stock in source warehouse");
        }

        // Remove from source warehouse
        self::updateStock(
            $product,
            $quantity,
            'OUT',
            $fromWarehouseId,
            null,
            null,
            $notes ? "Transfer to warehouse #{$toWarehouseId}: {$notes}" : "Transfer to warehouse #{$toWarehouseId}"
        );

        // Add to destination warehouse
        self::updateStock(
            $product,
            $quantity,
            'IN',
            $toWarehouseId,
            null,
            null,
            $notes ? "Transfer from warehouse #{$fromWarehouseId}: {$notes}" : "Transfer from warehouse #{$fromWarehouseId}"
        );
    }

    /**
     * Get stock value for a product
     */
    public static function getStockValue(Product $product, $warehouseId = null): float
    {
        $quantity = self::getStockQuantity($product, $warehouseId);
        return $quantity * $product->purchase_price;
    }
}

