<?php

namespace App\Services;

use App\Models\Setting;
use Carbon\Carbon;

class TrialService
{
    protected $trialDays = 20;
    protected $warningDays = 3; // تحذير قبل 3 أيام

    /**
     * الحصول على تاريخ بداية الفترة التجريبية
     */
    public function getStartDate(): ?Carbon
    {
        // الحصول مباشرة من قاعدة البيانات بدون Cache
        $setting = \App\Models\Setting::where('key', 'trial_start_date')->first();
        $startDate = $setting ? $setting->value : null;
        
        if (!$startDate) {
            // إذا لم يكن موجود، قم بإنشائه الآن
            $startDate = Carbon::now()->format('Y-m-d');
            Setting::set('trial_start_date', $startDate);
            // مسح Cache
            \Illuminate\Support\Facades\Cache::forget('setting.trial_start_date');
        }
        
        return Carbon::parse($startDate);
    }

    /**
     * الحصول على تاريخ انتهاء الفترة التجريبية
     */
    public function getEndDate(): Carbon
    {
        return $this->getStartDate()->copy()->addDays($this->trialDays);
    }

    /**
     * التحقق من انتهاء الفترة التجريبية
     */
    public function isExpired(): bool
    {
        return Carbon::now()->greaterThan($this->getEndDate());
    }

    /**
     * التحقق من قرب انتهاء الفترة (قبل 3 أيام)
     */
    public function isExpiringSoon(): bool
    {
        $endDate = $this->getEndDate();
        $warningDate = $endDate->copy()->subDays($this->warningDays);
        
        return Carbon::now()->greaterThanOrEqualTo($warningDate) && !$this->isExpired();
    }

    /**
     * الحصول على الأيام المتبقية
     */
    public function getDaysRemaining(): int
    {
        $endDate = $this->getEndDate();
        $now = Carbon::now();
        
        if ($now->greaterThan($endDate)) {
            return 0;
        }
        
        return $now->diffInDays($endDate) + 1;
    }

    /**
     * الحصول على معلومات الفترة التجريبية
     */
    public function getTrialInfo(): array
    {
        return [
            'start_date' => $this->getStartDate()->format('Y-m-d'),
            'end_date' => $this->getEndDate()->format('Y-m-d'),
            'days_remaining' => $this->getDaysRemaining(),
            'is_expired' => $this->isExpired(),
            'is_expiring_soon' => $this->isExpiringSoon(),
            'total_days' => $this->trialDays,
        ];
    }

    /**
     * إعادة تعيين الفترة التجريبية (للاستخدام من قبل المطور فقط)
     */
    public function resetTrial(): void
    {
        Setting::forget('trial_start_date');
    }

    /**
     * تعيين تاريخ البداية (للاستخدام من قبل المطور فقط)
     */
    public function setStartDate(int $daysAgo): void
    {
        $startDate = Carbon::now()->subDays($daysAgo);
        Setting::set('trial_start_date', $startDate->format('Y-m-d'));
        // مسح Cache
        \Illuminate\Support\Facades\Cache::forget('setting.trial_start_date');
    }
}

