@extends('layouts.master')

@section('title', trans('messages.new_inventory'))

@section('content')
<div class="page-header">
    <h2>{{ trans('messages.new_inventory') }}</h2>
    <a href="{{ route('inventories.index') }}" class="btn btn-secondary">{{ trans('messages.back') }}</a>
</div>

<div class="card">
    <div class="card-body">
        <form method="POST" action="{{ route('inventories.store') }}" id="inventory-form">
            @csrf
            <div class="row mb-3">
                <div class="col-md-4">
                    <label for="reference" class="form-label">{{ trans('messages.reference') }} *</label>
                    <input type="text" class="form-control" id="reference" name="reference" value="{{ $reference }}" required readonly>
                </div>
                <div class="col-md-4">
                    <label for="inventory_date" class="form-label">{{ trans('messages.date') }} *</label>
                    <input type="date" class="form-control" id="inventory_date" name="inventory_date" value="{{ date('Y-m-d') }}" required>
                </div>
            </div>

            <div class="mb-3">
                <label for="notes" class="form-label">{{ trans('messages.notes') }}</label>
                <textarea class="form-control" id="notes" name="notes" rows="3"></textarea>
            </div>

            <hr>

            <h5>{{ trans('messages.inventory_items') }}</h5>
            <div class="table-responsive mb-3">
                <table class="table" id="items-table">
                    <thead>
                        <tr>
                            <th>{{ trans('messages.product') }}</th>
                            <th>{{ trans('messages.system_quantity') }}</th>
                            <th>{{ trans('messages.physical_quantity') }}</th>
                            <th>{{ trans('messages.difference') }}</th>
                            <th>{{ trans('messages.notes') }}</th>
                            <th>{{ trans('messages.actions') }}</th>
                        </tr>
                    </thead>
                    <tbody id="items-tbody">
                        <tr class="item-row">
                            <td>
                                <select class="form-select product-select" name="items[0][product_id]" required>
                                    <option value="">{{ trans('messages.select_product') }}</option>
                                    @foreach($products as $product)
                                        <option value="{{ $product->id }}" data-stock="{{ $product->stock_quantity }}">
                                            {{ $product->name }} ({{ trans('messages.stock') }}: {{ $product->stock_quantity }})
                                        </option>
                                    @endforeach
                                </select>
                            </td>
                            <td>
                                <input type="number" class="form-control system-quantity" readonly value="0">
                            </td>
                            <td>
                                <input type="number" class="form-control physical-quantity" name="items[0][physical_quantity]" min="0" value="0" required>
                            </td>
                            <td>
                                <span class="difference-badge">0</span>
                            </td>
                            <td>
                                <input type="text" class="form-control" name="items[0][notes]" placeholder="{{ trans('messages.notes') }}">
                            </td>
                            <td>
                                <button type="button" class="btn btn-sm btn-danger remove-item">
                                    <i class="bi bi-trash"></i>
                                </button>
                            </td>
                        </tr>
                    </tbody>
                </table>
            </div>
            <button type="button" class="btn btn-sm btn-primary mb-3" id="add-item">
                <i class="bi bi-plus"></i> {{ trans('messages.add_item') }}
            </button>

            <div class="text-end">
                <button type="submit" class="btn btn-primary btn-lg">{{ trans('messages.create_inventory') }}</button>
            </div>
        </form>
    </div>
</div>

@push('scripts')
<script>
let itemIndex = 1;

// Add new item row
document.getElementById('add-item').addEventListener('click', function() {
    const tbody = document.getElementById('items-tbody');
    const newRow = document.createElement('tr');
    newRow.className = 'item-row';
    newRow.innerHTML = `
        <td>
            <select class="form-select product-select" name="items[${itemIndex}][product_id]" required>
                <option value="">{{ trans('messages.select_product') }}</option>
                @foreach($products as $product)
                    <option value="{{ $product->id }}" data-stock="{{ $product->stock_quantity }}">
                        {{ $product->name }} ({{ trans('messages.stock') }}: {{ $product->stock_quantity }})
                    </option>
                @endforeach
            </select>
        </td>
        <td>
            <input type="number" class="form-control system-quantity" readonly value="0">
        </td>
        <td>
            <input type="number" class="form-control physical-quantity" name="items[${itemIndex}][physical_quantity]" min="0" value="0" required>
        </td>
        <td>
            <span class="difference-badge">0</span>
        </td>
        <td>
            <input type="text" class="form-control" name="items[${itemIndex}][notes]" placeholder="{{ trans('messages.notes') }}">
        </td>
        <td>
            <button type="button" class="btn btn-sm btn-danger remove-item">
                <i class="bi bi-trash"></i>
            </button>
        </td>
    `;
    tbody.appendChild(newRow);
    itemIndex++;
    attachEventListeners(newRow);
});

// Remove item
document.addEventListener('click', function(e) {
    if (e.target.closest('.remove-item')) {
        const tbody = document.getElementById('items-tbody');
        const rows = tbody.querySelectorAll('tr.item-row');
        if (rows.length > 1) {
            e.target.closest('tr.item-row').remove();
        } else {
            alert('{{ trans("messages.at_least_one_item_required") }}');
        }
    }
});

// Calculate difference when product or quantity changes
document.addEventListener('change', function(e) {
    if (e.target.classList.contains('product-select') || e.target.classList.contains('physical-quantity')) {
        const row = e.target.closest('tr.item-row');
        if (row) {
            calculateDifference(row);
        }
    }
});

function calculateDifference(row) {
    const productSelect = row.querySelector('.product-select');
    const systemQuantityInput = row.querySelector('.system-quantity');
    const physicalQuantityInput = row.querySelector('.physical-quantity');
    const differenceBadge = row.querySelector('.difference-badge');

    if (productSelect.value) {
        const stock = parseInt(productSelect.options[productSelect.selectedIndex].dataset.stock || 0);
        systemQuantityInput.value = stock;
        
        const physicalQty = parseInt(physicalQuantityInput.value || 0);
        const difference = physicalQty - stock;
        
        differenceBadge.textContent = difference;
        differenceBadge.className = 'difference-badge badge ' + 
            (difference > 0 ? 'bg-success' : difference < 0 ? 'bg-danger' : 'bg-secondary');
    } else {
        systemQuantityInput.value = 0;
        differenceBadge.textContent = 0;
        differenceBadge.className = 'difference-badge badge bg-secondary';
    }
}

function attachEventListeners(row) {
    const productSelect = row.querySelector('.product-select');
    const physicalQuantityInput = row.querySelector('.physical-quantity');
    
    if (productSelect) {
        productSelect.addEventListener('change', function() {
            calculateDifference(row);
        });
    }
    
    if (physicalQuantityInput) {
        physicalQuantityInput.addEventListener('input', function() {
            calculateDifference(row);
        });
    }
}

// Attach listeners to initial row
document.querySelectorAll('.item-row').forEach(row => {
    attachEventListeners(row);
});
</script>
@endpush

@endsection

