@extends('layouts.master')

@section('title', trans('messages.create_invoice'))

@section('content')
@php
    // Get warehouse ID if multiple warehouses mode
    $warehouseId = null;
    if (\App\Services\StockManager::isMultipleWarehouses()) {
        $warehouseId = \App\Services\StockManager::getDefaultWarehouse()?->id;
    }
@endphp
<div class="page-header">
    <h2>{{ trans('messages.create_invoice') }}</h2>
    <a href="{{ route('invoices.index') }}" class="btn btn-secondary">{{ trans('messages.back') }}</a>
</div>

<div class="card">
    <div class="card-body">
        <form method="POST" action="{{ route('invoices.store') }}" id="invoice-form">
            @csrf
            <div class="row mb-3">
                <div class="col-md-6">
                    <label for="customer_id" class="form-label">{{ trans('messages.customer') }}</label>
                    <select class="form-select" id="customer_id" name="customer_id">
                        <option value="">{{ trans('messages.walk_in_customer') }}</option>
                        @foreach($customers as $customer)
                            <option value="{{ $customer->id }}">{{ $customer->name }}</option>
                        @endforeach
                    </select>
                </div>
                <div class="col-md-6">
                    <label for="customer_name" class="form-label">{{ trans('messages.customer_name') }} ({{ trans('messages.walk_in_customer') }})</label>
                    <input type="text" class="form-control" id="customer_name" name="customer_name" placeholder="{{ trans('messages.walk_in_customer') }}">
                </div>
            </div>

            <hr>

            <h5>{{ trans('messages.items') }}</h5>
            <div class="table-responsive mb-3">
                <table class="table" id="items-table">
                    <thead>
                        <tr>
                            <th>{{ trans('messages.products') }}</th>
                            <th>{{ trans('messages.quantity') }}</th>
                            <th>{{ trans('messages.unit_price') }}</th>
                            <th>{{ trans('messages.tax') }} %</th>
                            <th>{{ trans('messages.discount') }}</th>
                            <th>{{ trans('messages.total') }}</th>
                            <th>{{ trans('messages.actions') }}</th>
                        </tr>
                    </thead>
                    <tbody id="items-tbody">
                        <tr class="item-row">
                            <td>
                                <select id="product-select-0" class="form-select product-select" name="items[0][product_id]" required>
                                    <option value="">{{ trans('messages.select_product') }}</option>
                                    @foreach($products as $product)
                                        @php
                                            $stockQuantity = \App\Services\StockManager::getStockQuantity($product, $warehouseId);
                                        @endphp
                                        <option value="{{ $product->id }}" data-price="{{ $product->selling_price }}" data-tax="{{ $product->tax_rate }}" data-stock="{{ $stockQuantity }}">{{ $product->name }} ({{ trans('messages.stock') }}: {{ $stockQuantity }})</option>
                                    @endforeach
                                </select>
                            </td>
                            <td><input type="number" id="quantity-0" class="form-control quantity" name="items[0][quantity]" min="1" value="1" required></td>
                            <td><input type="number" step="0.01" id="unit-price-0" class="form-control unit-price" name="items[0][unit_price]" required></td>
                            <td><input type="number" step="0.01" id="tax-rate-0" class="form-control tax-rate" name="items[0][tax_rate]" value="0"></td>
                            <td><input type="number" step="0.01" id="discount-0" class="form-control discount" name="items[0][discount]" value="0"></td>
                            <td><span class="item-total">0.00</span> {{ currency_symbol() }}</td>
                            <td><button type="button" id="remove-item-0" class="btn btn-sm btn-danger remove-item"><i class="bi bi-trash"></i></button></td>
                        </tr>
                    </tbody>
                </table>
            </div>
            <button type="button" class="btn btn-sm btn-primary mb-3" id="add-item"><i class="bi bi-plus"></i> {{ trans('messages.add_item') }}</button>

            <div class="row">
                <div class="col-md-6">
                    <div class="mb-3">
                        <label for="discount" class="form-label">{{ trans('messages.invoice_discount') }}</label>
                        <input type="number" step="0.01" class="form-control" id="invoice-discount" name="discount" value="0">
                    </div>
                </div>
                <div class="col-md-6">
                    <div class="mb-3">
                        <label for="notes" class="form-label">{{ trans('messages.notes') }}</label>
                        <textarea class="form-control" id="notes" name="notes" rows="3"></textarea>
                    </div>
                </div>
            </div>

            <div class="text-end">
                <h4>{{ trans('messages.total') }}: <span id="grand-total">0.00</span> {{ currency_symbol() }}</h4>
                <button type="submit" class="btn btn-primary btn-lg">{{ trans('messages.create_invoice') }}</button>
            </div>
        </form>
    </div>
</div>

@push('scripts')
<script>
let itemIndex = 1;
const currency = '{{ currency_symbol() }}';

$('#add-item').click(function() {
    const newRow = `
        <tr class="item-row">
            <td>
                <select id="product-select-${itemIndex}" class="form-select product-select" name="items[${itemIndex}][product_id]" required>
                    <option value="">{{ trans('messages.select_product') }}</option>
                    @foreach($products as $product)
                        @php
                            $stockQuantity = \App\Services\StockManager::getStockQuantity($product, $warehouseId);
                        @endphp
                        <option value="{{ $product->id }}" data-price="{{ $product->selling_price }}" data-tax="{{ $product->tax_rate }}" data-stock="{{ $stockQuantity }}">{{ $product->name }} ({{ trans('messages.stock') }}: {{ $stockQuantity }})</option>
                    @endforeach
                </select>
            </td>
            <td><input type="number" id="quantity-${itemIndex}" class="form-control quantity" name="items[${itemIndex}][quantity]" min="1" value="1" required></td>
            <td><input type="number" step="0.01" id="unit-price-${itemIndex}" class="form-control unit-price" name="items[${itemIndex}][unit_price]" required></td>
            <td><input type="number" step="0.01" id="tax-rate-${itemIndex}" class="form-control tax-rate" name="items[${itemIndex}][tax_rate]" value="0"></td>
            <td><input type="number" step="0.01" id="discount-${itemIndex}" class="form-control discount" name="items[${itemIndex}][discount]" value="0"></td>
            <td><span class="item-total">0.00</span> ${currency}</td>
            <td><button type="button" id="remove-item-${itemIndex}" class="btn btn-sm btn-danger remove-item"><i class="bi bi-trash"></i></button></td>
        </tr>
    `;
    $('#items-tbody').append(newRow);
    itemIndex++;
});

// Remove item handler - using both jQuery and vanilla JS for compatibility
$(document).on('click', '.remove-item', function(e) {
    e.preventDefault();
    e.stopPropagation();
    
    const tbody = $('#items-tbody');
    const rows = tbody.find('tr.item-row');
    
    // Prevent removing the last row - must have at least one item
    if (rows.length > 1) {
        $(this).closest('tr.item-row').remove();
        calculateTotal();
    } else {
        alert('{{ trans("messages.at_least_one_item_required") }}');
    }
});

// Fallback using vanilla JavaScript if jQuery fails
document.addEventListener('click', function(e) {
    if (e.target.closest('.remove-item')) {
        e.preventDefault();
        e.stopPropagation();
        
        const row = e.target.closest('tr.item-row');
        const tbody = document.getElementById('items-tbody');
        const rows = tbody.querySelectorAll('tr.item-row');
        
        if (rows.length > 1 && row) {
            row.remove();
            if (typeof calculateTotal === 'function') {
                calculateTotal();
            }
        } else {
            alert('{{ trans("messages.at_least_one_item_required") }}');
        }
    }
});

$(document).on('change', '.product-select', function() {
    const option = $(this).find('option:selected');
    const row = $(this).closest('tr');
    row.find('.unit-price').val(option.data('price') || 0);
    row.find('.tax-rate').val(option.data('tax') || 0);
    calculateRowTotal(row);
});

$(document).on('input', '.quantity, .unit-price, .tax-rate, .discount, #invoice-discount', function() {
    const row = $(this).closest('tr');
    if (row.hasClass('item-row')) {
        calculateRowTotal(row);
    } else {
        calculateTotal();
    }
});

function calculateRowTotal(row) {
    const quantity = parseFloat(row.find('.quantity').val()) || 0;
    const unitPrice = parseFloat(row.find('.unit-price').val()) || 0;
    const taxRate = parseFloat(row.find('.tax-rate').val()) || 0;
    const discount = parseFloat(row.find('.discount').val()) || 0;
    
    const subtotal = quantity * unitPrice;
    const afterDiscount = subtotal - discount;
    const tax = afterDiscount * (taxRate / 100);
    const total = afterDiscount + tax;
    
    row.find('.item-total').text(total.toFixed(2));
    calculateTotal();
}

function calculateTotal() {
    let total = 0;
    $('.item-row').each(function() {
        total += parseFloat($(this).find('.item-total').text()) || 0;
    });
    
    const invoiceDiscount = parseFloat($('#invoice-discount').val()) || 0;
    total -= invoiceDiscount;
    
    $('#grand-total').text(total.toFixed(2));
}
</script>
@endpush
@endsection
