@extends('layouts.master')

@section('title', trans('messages.product_details'))

@section('content')
<div class="page-header">
    <h2>{{ trans('messages.product_details') }}</h2>
    <div>
        @can('products.update')
            <a href="{{ route('products.edit', $product) }}" class="btn btn-warning"><i class="bi bi-pencil"></i> {{ trans('messages.edit') }}</a>
        @endcan
        <a href="{{ route('products.index') }}" class="btn btn-secondary">{{ trans('messages.back') }}</a>
    </div>
</div>

<div class="row">
    <div class="col-md-8">
        <div class="card">
            <div class="card-body">
                <h4>{{ $product->name }}</h4>
                @if($product->name_ar)
                    <p class="text-muted">{{ $product->name_ar }}</p>
                @endif
                <hr>
                <div class="row">
                    <div class="col-md-6">
                        <div class="mb-3">
                            <strong>{{ trans('messages.sku') }}:</strong> {{ $product->sku }}
                            <button type="button" class="btn btn-sm btn-outline-primary ms-2" data-bs-toggle="modal" data-bs-target="#barcodeModal" onclick="showBarcodeModal('sku', '{{ $product->sku }}')">
                                <i class="bi bi-upc-scan"></i> {{ trans('messages.print_barcode') }}
                            </button>
                        </div>
                        @if($product->barcode)
                            <div class="mb-3">
                                <strong>{{ trans('messages.barcode') }}:</strong> {{ $product->barcode }}
                                <button type="button" class="btn btn-sm btn-outline-primary ms-2" data-bs-toggle="modal" data-bs-target="#barcodeModal" onclick="showBarcodeModal('barcode', '{{ $product->barcode }}')">
                                    <i class="bi bi-upc-scan"></i> {{ trans('messages.print_barcode') }}
                                </button>
                            </div>
                        @endif
                        <p><strong>{{ trans('messages.category') }}:</strong> {{ $product->category?->name ?? 'N/A' }}</p>
                        @if($product->brand)
                            <p><strong>{{ trans('messages.brand') }}:</strong> {{ $product->brand }}</p>
                        @endif
                        <p><strong>{{ trans('messages.unit') }}:</strong> {{ $product->unit }}</p>
                    </div>
                    <div class="col-md-6">
                        <p><strong>{{ trans('messages.purchase_price') }}:</strong> {{ format_currency($product->purchase_price) }}</p>
                        <p><strong>{{ trans('messages.selling_price') }}:</strong> {{ format_currency($product->selling_price) }}</p>
                        <p><strong>{{ trans('messages.tax_rate') }}:</strong> {{ $product->tax_rate }}%</p>
                        <p><strong>{{ trans('messages.stock') }}:</strong> 
                            <span class="badge bg-{{ $product->isLowStock() ? 'danger' : 'success' }}">{{ $product->stock_quantity }}</span>
                        </p>
                        <p><strong>{{ trans('messages.reorder_level') }}:</strong> {{ $product->reorder_level }}</p>
                    </div>
                </div>
                @if($product->description)
                    <hr>
                    <p><strong>{{ trans('messages.description') }}:</strong></p>
                    <p>{{ $product->description }}</p>
                @endif
            </div>
        </div>

        <!-- Warehouse Stock (Multiple Warehouses Mode) -->
        @if(\App\Services\StockManager::isMultipleWarehouses() && $product->warehouses->count() > 0)
        <div class="card mt-3">
            <div class="card-header">
                <h5 class="mb-0"><i class="bi bi-boxes"></i> {{ trans('messages.stock_in_warehouses') }}</h5>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-sm">
                        <thead>
                            <tr>
                                <th>{{ trans('messages.warehouse') }}</th>
                                <th class="text-end">{{ trans('messages.quantity') }}</th>
                                <th class="text-end">{{ trans('messages.reorder_level') }}</th>
                                <th class="text-center">{{ trans('messages.status') }}</th>
                            </tr>
                        </thead>
                        <tbody>
                            @php
                                $defaultWarehouse = \App\Services\StockManager::getDefaultWarehouse();
                            @endphp
                            @foreach($product->warehouses as $warehouse)
                            @php
                                $pivot = $product->warehouses()->where('warehouses.id', $warehouse->id)->first();
                                $quantity = $pivot ? $pivot->pivot->quantity : 0;
                                $reorderLevel = $pivot ? $pivot->pivot->reorder_level : $product->reorder_level;
                                $isLowStock = $quantity <= $reorderLevel;
                                $isDefault = $defaultWarehouse && $defaultWarehouse->id == $warehouse->id;
                            @endphp
                            <tr class="{{ $isDefault ? 'table-primary' : '' }}">
                                <td>
                                    <strong>{{ app()->getLocale() === 'ar' && $warehouse->name_ar ? $warehouse->name_ar : $warehouse->name }}</strong>
                                    @if($isDefault)
                                        <span class="badge bg-info ms-2">{{ trans('messages.default') }}</span>
                                    @endif
                                </td>
                                <td class="text-end">
                                    <span class="badge bg-{{ $isLowStock ? 'danger' : 'success' }}">{{ number_format($quantity) }}</span>
                                </td>
                                <td class="text-end">{{ number_format($reorderLevel) }}</td>
                                <td class="text-center">
                                    @if($isLowStock)
                                        <span class="badge bg-warning">{{ trans('messages.low_stock') }}</span>
                                    @else
                                        <span class="badge bg-success">{{ trans('messages.in_stock') }}</span>
                                    @endif
                                </td>
                            </tr>
                            @endforeach
                        </tbody>
                        <tfoot>
                            <tr class="table-secondary">
                                <td><strong>{{ trans('messages.total') }}</strong></td>
                                <td class="text-end">
                                    <strong>{{ number_format($product->warehouses->sum(function($w) use ($product) {
                                        $pivot = $product->warehouses()->where('warehouses.id', $w->id)->first();
                                        return $pivot ? $pivot->pivot->quantity : 0;
                                    })) }}</strong>
                                </td>
                                <td colspan="2"></td>
                            </tr>
                        </tfoot>
                    </table>
                </div>
            </div>
        </div>
        @endif

        <!-- Product Batches -->
        @if($product->batches->count() > 0)
        <div class="card mt-3">
            <div class="card-header">
                <h5 class="mb-0"><i class="bi bi-box-seam"></i> {{ trans('messages.product_batches') }}</h5>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-sm">
                        <thead>
                            <tr>
                                <th>{{ trans('messages.batch_number') }}</th>
                                <th>{{ trans('messages.supplier') }}</th>
                                <th>{{ trans('messages.purchase_invoice') }}</th>
                                <th>{{ trans('messages.purchase_date') }}</th>
                                <th>{{ trans('messages.expiry_date') }}</th>
                                <th class="text-end">{{ trans('messages.initial_quantity') }}</th>
                                <th class="text-end">{{ trans('messages.current_quantity') }}</th>
                                <th class="text-end">{{ trans('messages.purchase_price') }}</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($product->batches->sortByDesc('purchase_date') as $batch)
                            <tr class="{{ $batch->isExpired() ? 'table-danger' : ($batch->isExpiringSoon() ? 'table-warning' : '') }}">
                                <td>
                                    <strong>{{ $batch->batch_number }}</strong>
                                </td>
                                <td>
                                    <a href="{{ route('suppliers.show', $batch->supplier) }}">{{ $batch->supplier->name }}</a>
                                </td>
                                <td>
                                    <a href="{{ route('purchase-invoices.show', $batch->purchaseInvoice) }}">
                                        {{ $batch->purchaseInvoice->invoice_number }}
                                    </a>
                                </td>
                                <td>{{ $batch->purchase_date->format('Y-m-d') }}</td>
                                <td>
                                    @if($batch->expiry_date)
                                        <span class="{{ $batch->isExpired() ? 'text-danger' : ($batch->isExpiringSoon() ? 'text-warning' : '') }}">
                                            {{ $batch->expiry_date->format('Y-m-d') }}
                                        </span>
                                        @if($batch->isExpired())
                                            <span class="badge bg-danger ms-1">{{ trans('messages.expired') }}</span>
                                        @elseif($batch->isExpiringSoon())
                                            <span class="badge bg-warning ms-1">{{ trans('messages.expiring_soon') }}</span>
                                        @endif
                                    @else
                                        <span class="text-muted">-</span>
                                    @endif
                                </td>
                                <td class="text-end">{{ number_format($batch->initial_quantity) }}</td>
                                <td class="text-end">
                                    <strong>{{ number_format($batch->current_quantity) }}</strong>
                                    <small class="text-muted d-block">({{ number_format($batch->remaining_percentage, 1) }}%)</small>
                                </td>
                                <td class="text-end">{{ format_currency($batch->purchase_price) }}</td>
                            </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        @endif
    </div>
    <div class="col-md-4">
        <div class="card mb-3">
            <div class="card-body text-center">
                @if($product->image)
                    <img src="{{ storage_url($product->image) }}" alt="{{ $product->name }}" class="img-fluid" style="max-height: 300px;">
                @else
                    <div class="d-flex align-items-center justify-content-center bg-light" style="min-height: 300px;">
                        <img src="{{ asset('assets/images/no-product.svg') }}" alt="{{ trans('messages.no_image') }}" style="max-width: 150px; opacity: 0.5;">
                    </div>
                @endif
            </div>
        </div>
    </div>
</div>

<!-- Barcode Print Modal -->
<div class="modal fade" id="barcodeModal" tabindex="-1" aria-labelledby="barcodeModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="barcodeModalLabel">{{ trans('messages.print_barcode') }}</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div class="mb-3">
                    <label for="printQuantity" class="form-label">{{ trans('messages.print_quantity') }}</label>
                    <input type="number" class="form-control" id="printQuantity" value="1" min="1" max="100">
                </div>
                <div id="barcodeContainer" class="text-center barcode-container">
                    <!-- Barcodes will be generated here -->
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">{{ trans('messages.close') }}</button>
                <button type="button" class="btn btn-success" onclick="exportBarcodesAsPNG()">
                    <i class="bi bi-download"></i> {{ trans('messages.export_png') }}
                </button>
                <button type="button" class="btn btn-primary" onclick="printBarcodes()">
                    <i class="bi bi-printer"></i> {{ trans('messages.print') }}
                </button>
            </div>
        </div>
    </div>
</div>

@push('styles')
<style>
    .barcode-item {
        display: inline-block;
        margin: 10px;
        padding: 15px;
        border: 1px solid #ddd;
        border-radius: 5px;
        text-align: center;
    }
    .barcode-item svg {
        display: block;
        margin: 0 auto;
        width: 100%;
        max-width: 100%;
    }
    .barcode-label {
        margin-top: 10px;
        font-weight: bold;
    }
    @media print {
        @page {
            margin: 10mm;
        }
        body * {
            visibility: hidden !important;
        }
        .modal, .modal * {
            visibility: visible !important;
        }
        .modal {
            position: absolute !important;
            left: 0 !important;
            top: 0 !important;
            width: 100% !important;
            height: 100% !important;
            margin: 0 !important;
            padding: 0 !important;
            display: block !important;
            border: none !important;
        }
        .modal-content {
            border: none !important;
            box-shadow: none !important;
        }
        .modal-dialog {
            max-width: 100% !important;
            margin: 0 !important;
            padding: 20px !important;
        }
        .modal-header, .modal-footer, .btn, #printQuantity, .mb-3 {
            display: none !important;
            visibility: hidden !important;
        }
        .barcode-item {
            page-break-inside: avoid;
            margin: 0 !important;
            padding: 10px !important;
            visibility: visible !important;
            border: none !important;
            width: 100% !important;
            display: block !important;
            box-sizing: border-box !important;
        }
        .barcode-container {
            display: block !important;
            width: 100% !important;
            visibility: visible !important;
            margin: 0 !important;
            padding: 0 !important;
        }
        .barcode-product-name {
            visibility: visible !important;
            font-size: 48px !important;
            font-weight: bold !important;
            margin-bottom: 20px !important;
            text-align: center !important;
            line-height: 1.4 !important;
        }
        .barcode-label {
            visibility: visible !important;
            font-size: 16px !important;
            margin-top: 15px !important;
        }
        .barcode-svg {
            visibility: visible !important;
            width: 100% !important;
            max-width: 100% !important;
            height: auto !important;
        }
        .barcode-svg svg {
            width: 100% !important;
            max-width: 100% !important;
            height: auto !important;
        }
        footer, .footer {
            display: none !important;
            visibility: hidden !important;
        }
        fieldset {
            border: none !important;
            padding: 0 !important;
            margin: 0 !important;
        }
    }
</style>
@endpush

@push('scripts')
<script src="{{ asset('lib/js/JsBarcode.all.min.js') }}"></script>
<script>
    let currentBarcodeType = '';
    let currentBarcodeValue = '';
    let productSku = '{{ $product->sku }}';
    let productBarcode = '{{ $product->barcode ?? '' }}';
    let barcodeFormat = '{{ \App\Models\Setting::get("barcode_format", "CODE128") }}';
    let showProductName = {{ \App\Models\Setting::get("show_product_name_in_barcode", "0") === "1" ? "true" : "false" }};
    let productName = '{{ addslashes($product->name) }}';

    function showBarcodeModal(type, value) {
        currentBarcodeType = type;
        currentBarcodeValue = value;
        generateBarcodes();
    }

    function generateBarcodes() {
        const container = document.getElementById('barcodeContainer');
        const quantity = parseInt(document.getElementById('printQuantity').value) || 1;
        container.innerHTML = '';

        for (let i = 0; i < quantity; i++) {
            const barcodeDiv = document.createElement('div');
            barcodeDiv.className = 'barcode-item';
            
            // Create SVG for barcode
            const svg = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
            svg.id = 'barcode-' + i;
            svg.setAttribute('class', 'barcode-svg');
            
            barcodeDiv.appendChild(svg);
            
            // Create product name if enabled
            if (showProductName && productName) {
                const productNameDiv = document.createElement('div');
                productNameDiv.className = 'barcode-product-name';
                productNameDiv.textContent = productName;
                barcodeDiv.appendChild(productNameDiv);
            }
            
            // Create label
            const label = document.createElement('div');
            label.className = 'barcode-label';
            label.textContent = currentBarcodeValue;
            barcodeDiv.appendChild(label);
            
            container.appendChild(barcodeDiv);
            
            // Generate barcode using JsBarcode with format from settings
            try {
                JsBarcode('#barcode-' + i, currentBarcodeValue, {
                    format: barcodeFormat,
                    width: 2,
                    height: 80,
                    displayValue: true,
                    fontSize: 14,
                    margin: 10
                });
            } catch (e) {
                console.error('Error generating barcode:', e);
                // Fallback to CODE128 if format fails
                try {
                    JsBarcode('#barcode-' + i, currentBarcodeValue, {
                        format: "CODE128",
                        width: 2,
                        height: 80,
                        displayValue: true,
                        fontSize: 14,
                        margin: 10
                    });
                } catch (e2) {
                    console.error('Error generating barcode with fallback:', e2);
                }
            }
        }
    }

    function printBarcodes() {
        window.print();
    }

    function exportBarcodesAsPNG() {
        const container = document.getElementById('barcodeContainer');
        const barcodeItems = container.querySelectorAll('.barcode-item');
        
        if (barcodeItems.length === 0) {
            alert('{{ trans("messages.no_barcodes_to_export") }}');
            return;
        }

        // Export each barcode as PNG with delay to avoid browser blocking multiple downloads
        barcodeItems.forEach((item, index) => {
            setTimeout(() => {
                const svg = item.querySelector('svg');
                if (!svg) return;

                try {
                    // Get product name if enabled
                    const productNameDiv = item.querySelector('.barcode-product-name');
                    const productName = showProductName && productNameDiv ? productNameDiv.textContent : '';
                    
                    // Get barcode label
                    const labelDiv = item.querySelector('.barcode-label');
                    const barcodeLabel = labelDiv ? labelDiv.textContent : currentBarcodeValue;
                    
                    // Get SVG dimensions and scale up
                    const svgRect = svg.getBoundingClientRect();
                    const scale = 3; // Scale up 3x for better quality
                    const svgWidth = (svgRect.width || 400) * scale;
                    const svgHeight = (svgRect.height || 100) * scale;
                    
                    // Calculate total height including product name and label
                    let totalHeight = svgHeight;
                    let nameHeight = 0;
                    let labelHeight = 0;
                    const nameFontSize = 24; // Reduced from 48px
                    
                    if (productName) {
                        nameHeight = nameFontSize * scale * 1.5; // Calculate based on font size
                    }
                    if (barcodeLabel) {
                        labelHeight = 48; // 16px * 3 scale
                    }
                    
                    totalHeight = nameHeight + svgHeight + labelHeight + (40 * scale); // Add padding
                    
                    // Create canvas with larger size
                    const canvas = document.createElement('canvas');
                    const ctx = canvas.getContext('2d');
                    canvas.width = svgWidth;
                    canvas.height = totalHeight;
                    
                    // Set white background
                    ctx.fillStyle = '#FFFFFF';
                    ctx.fillRect(0, 0, canvas.width, canvas.height);
                    
                    let yOffset = 20 * scale;
                    
                    // Draw product name if enabled
                    if (productName) {
                        ctx.fillStyle = '#000000';
                        ctx.font = 'bold ' + (nameFontSize * scale) + 'px Arial';
                        ctx.textAlign = 'center';
                        ctx.textBaseline = 'top';
                        const nameLines = wrapText(ctx, productName, canvas.width - (40 * scale));
                        nameLines.forEach((line, i) => {
                            ctx.fillText(line, canvas.width / 2, yOffset + (i * (nameFontSize * scale * 1.2)));
                        });
                        yOffset += nameHeight + (20 * scale);
                    }
                    
                    // Clone SVG and scale it
                    const clonedSvg = svg.cloneNode(true);
                    clonedSvg.setAttribute('width', svgWidth);
                    clonedSvg.setAttribute('height', svgHeight);
                    clonedSvg.setAttribute('xmlns', 'http://www.w3.org/2000/svg');
                    
                    // Get SVG data as base64
                    const svgData = new XMLSerializer().serializeToString(clonedSvg);
                    const svgBase64 = 'data:image/svg+xml;base64,' + btoa(unescape(encodeURIComponent(svgData)));
                    
                    // Create image for barcode
                    const img = new Image();
                    img.crossOrigin = 'anonymous';
                    
                    img.onload = function() {
                        // Draw barcode SVG on canvas
                        ctx.drawImage(img, 0, yOffset, svgWidth, svgHeight);
                        
                        yOffset += svgHeight + (20 * scale);
                        
                        // Draw barcode label
                        if (barcodeLabel) {
                            ctx.fillStyle = '#000000';
                            ctx.font = 'bold ' + (16 * scale) + 'px Arial';
                            ctx.textAlign = 'center';
                            ctx.textBaseline = 'top';
                            ctx.fillText(barcodeLabel, canvas.width / 2, yOffset);
                        }
                        
                        // Convert to PNG and download
                        canvas.toBlob(function(blob) {
                            if (blob) {
                                const link = document.createElement('a');
                                link.download = 'barcode-' + currentBarcodeValue.replace(/[^a-zA-Z0-9]/g, '_') + '-' + (index + 1) + '.png';
                                link.href = URL.createObjectURL(blob);
                                document.body.appendChild(link);
                                link.click();
                                document.body.removeChild(link);
                                URL.revokeObjectURL(link.href);
                            }
                        }, 'image/png');
                    };
                    
                    img.onerror = function() {
                        console.error('Error loading SVG image');
                    };
                    
                    img.src = svgBase64;
                } catch (e) {
                    console.error('Error exporting barcode:', e);
                }
            }, index * 200); // Delay each download by 200ms
        });
    }
    
    function wrapText(context, text, maxWidth) {
        const words = text.split(' ');
        const lines = [];
        let currentLine = words[0];
        
        for (let i = 1; i < words.length; i++) {
            const word = words[i];
            const width = context.measureText(currentLine + ' ' + word).width;
            if (width < maxWidth) {
                currentLine += ' ' + word;
            } else {
                lines.push(currentLine);
                currentLine = word;
            }
        }
        lines.push(currentLine);
        return lines;
    }

    // Regenerate barcodes when quantity changes
    document.addEventListener('DOMContentLoaded', function() {
        const printQuantity = document.getElementById('printQuantity');
        if (printQuantity) {
            printQuantity.addEventListener('change', function() {
                if (currentBarcodeValue) {
                    generateBarcodes();
                }
            });
        }
    });
</script>
@endpush
@endsection
