@extends('layouts.master')

@section('title', trans('messages.purchase_orders'))

@section('content')
<div class="page-header">
    <h2>{{ trans('messages.purchase_orders') }}</h2>
    <div>
        <a href="{{ route('purchase-orders.create') }}" class="btn btn-primary">
            <i class="bi bi-plus-circle"></i> {{ trans('messages.create_purchase_order') }}
        </a>
    </div>
</div>

<div class="card">
    <div class="card-body">
        @can('purchase-orders.bulk_delete')
        <div class="mb-3 d-flex gap-2">
            <button type="button" id="delete-selected-btn" class="btn btn-danger" disabled onclick="bulkDeletePurchaseOrders()">
                <i class="bi bi-trash"></i> {{ trans('messages.delete_selected') }}
            </button>
        </div>
        @endcan

        <form method="GET" action="{{ route('purchase-orders.index') }}" class="mb-3">
            <div class="row g-3">
                <div class="col-md-4">
                    <input type="text" name="search" class="form-control" placeholder="{{ trans('messages.search') }}..." value="{{ request('search') }}">
                </div>
                <div class="col-md-3">
                    <select name="status" class="form-select">
                        <option value="">{{ trans('messages.all_statuses') }}</option>
                        <option value="draft" {{ request('status') == 'draft' ? 'selected' : '' }}>{{ trans('messages.draft') }}</option>
                        <option value="pending" {{ request('status') == 'pending' ? 'selected' : '' }}>{{ trans('messages.pending') }}</option>
                        <option value="approved" {{ request('status') == 'approved' ? 'selected' : '' }}>{{ trans('messages.approved') }}</option>
                        <option value="completed" {{ request('status') == 'completed' ? 'selected' : '' }}>{{ trans('messages.completed') }}</option>
                    </select>
                </div>
                <div class="col-md-2">
                    <button type="submit" class="btn btn-secondary w-100">{{ trans('messages.filter') }}</button>
                </div>
            </div>
        </form>

        <div class="table-responsive">
            <table class="table table-hover">
                <thead>
                    <tr>
                        @can('purchase-orders.bulk_delete')
                        <th style="width: 40px;">
                            <input type="checkbox" id="select-all-orders" title="{{ trans('messages.select_all') }}">
                        </th>
                        @endcan
                        <th>{{ trans('messages.order_number') }}</th>
                        <th>{{ trans('messages.supplier') }}</th>
                        <th>{{ trans('messages.total_amount') }}</th>
                        <th>{{ trans('messages.expected_delivery_date') }}</th>
                        <th>{{ trans('messages.status') }}</th>
                        <th>{{ trans('messages.generation_type') }}</th>
                        <th>{{ trans('messages.created_at') }}</th>
                        <th>{{ trans('messages.actions') }}</th>
                    </tr>
                </thead>
                <tbody>
                    @forelse($orders as $order)
                    <tr>
                        @can('purchase-orders.bulk_delete')
                        <td>
                            @if($order->isDraft())
                                <input type="checkbox" class="order-checkbox" value="{{ $order->id }}">
                            @endif
                        </td>
                        @endcan
                        <td><a href="{{ route('purchase-orders.show', $order) }}">{{ $order->order_number }}</a></td>
                        <td>{{ $order->supplier?->name ?? '-' }}</td>
                        <td>{{ format_currency($order->total_amount) }}</td>
                        <td>{{ $order->expected_delivery_date ? $order->expected_delivery_date->format('Y-m-d') : '-' }}</td>
                        <td>
                            @php
                                $badgeClass = match($order->status) {
                                    'draft' => 'secondary',
                                    'pending' => 'warning',
                                    'approved' => 'success',
                                    'completed' => 'primary',
                                    'rejected' => 'danger',
                                    'cancelled' => 'dark',
                                    default => 'secondary'
                                };
                            @endphp
                            <span class="badge bg-{{ $badgeClass }}">{{ trans('messages.' . $order->status) }}</span>
                        </td>
                        <td>
                            <span class="badge bg-{{ $order->generation_type === 'auto' ? 'info' : 'secondary' }}">
                                {{ trans('messages.' . $order->generation_type) }}
                            </span>
                        </td>
                        <td>{{ $order->created_at->format('Y-m-d') }}</td>
                        <td>
                            <a href="{{ route('purchase-orders.show', $order) }}" class="btn btn-sm btn-info">{{ trans('messages.view') }}</a>
                            @if($order->isDraft())
                                <a href="{{ route('purchase-orders.edit', $order) }}" class="btn btn-sm btn-warning">{{ trans('messages.edit') }}</a>
                            @endif
                            @if($order->status === 'pending')
                                <form action="{{ route('purchase-orders.approve', $order) }}" method="POST" class="d-inline">
                                    @csrf
                                    <button type="submit" class="btn btn-sm btn-success">{{ trans('messages.approve') }}</button>
                                </form>
                            @endif
                            @if($order->isDraft())
                                <form action="{{ route('purchase-orders.destroy', $order) }}" method="POST" class="d-inline" onsubmit="return confirm('{{ trans('messages.confirm_delete') }}')">
                                    @csrf
                                    @method('DELETE')
                                    <button type="submit" class="btn btn-sm btn-danger">{{ trans('messages.delete') }}</button>
                                </form>
                            @endif
                        </td>
                    </tr>
                    @empty
                    <tr>
                        <td colspan="{{ auth()->user()->can('purchase-orders.bulk_delete') ? '9' : '8' }}" class="text-center">{{ trans('messages.no_purchase_orders_found') }}</td>
                    </tr>
                    @endforelse
                </tbody>
            </table>
        </div>

        <div class="mt-3">
            {{ $orders->links() }}
        </div>
    </div>
</div>

@can('purchase-orders.bulk_delete')
@push('scripts')
<script>
document.getElementById('select-all-orders')?.addEventListener('change', function() {
    const checkboxes = document.querySelectorAll('.order-checkbox');
    checkboxes.forEach(cb => cb.checked = this.checked);
    updateDeleteButton();
});

document.querySelectorAll('.order-checkbox').forEach(checkbox => {
    checkbox.addEventListener('change', updateDeleteButton);
});

function updateDeleteButton() {
    const checked = document.querySelectorAll('.order-checkbox:checked').length;
    const btn = document.getElementById('delete-selected-btn');
    if (btn) {
        btn.disabled = checked === 0;
    }
}

function bulkDeletePurchaseOrders() {
    const selected = Array.from(document.querySelectorAll('.order-checkbox:checked')).map(cb => cb.value);
    if (selected.length === 0) {
        alert('{{ trans('messages.no_items_selected') }}');
        return;
    }
    
    if (confirm('{{ trans('messages.confirm_bulk_delete_message') }}: ' + selected.length + ' {{ trans('messages.items') }}?')) {
        const form = document.createElement('form');
        form.method = 'POST';
        form.action = '{{ route('purchase-orders.bulk-delete') }}';
        
        // Add CSRF token
        const csrfToken = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content');
        const csrfInput = document.createElement('input');
        csrfInput.type = 'hidden';
        csrfInput.name = '_token';
        csrfInput.value = csrfToken || '{{ csrf_token() }}';
        form.appendChild(csrfInput);
        
        // Add selected IDs
        selected.forEach(id => {
            const input = document.createElement('input');
            input.type = 'hidden';
            input.name = 'selected_ids[]';
            input.value = id;
            form.appendChild(input);
        });
        
        document.body.appendChild(form);
        form.submit();
    }
}
</script>
@endpush
@endcan
@endsection

